// this code emulate a Wang 3320 terminal controller connected to a
// Wang 3315 terminal, which is a rebadged ASR33 TTY.
//
// This specific file emulates just the punch part of the TTY.
//
// TODO:
//   *) currently, the punch writes each byte received from the printer.
//      this doesn't simulate overstriking, which would require a R/M/W.

#include <fstream>
#include <string.h>     // needed for strdup

#include "system3300.h"
#include "term3315.h"
#include "term3315_impl.h"
#include "cpu3300.h"
#include "UiSystem.h"
#include "UiTerm3315.h"

extern FILE *g_outf;


// ======================================================================
// ======================================================================

term3315_pun::term3315_pun(int termnum, UiTerm3315 *ui) :
    m_termid(termnum),
    m_ui(ui),
    m_kbd(0),                   // this gets bound later
    m_tape_xon(ctl_xoff),
    m_tape_punch_enabled(false),
    m_kbd_busy(false)
{
    ASSERT(termnum >= 0 && termnum < 16);

    for(int u=0; u<2; u++)
        m_tape_filename[u] = 0;
}


// destructor doesn't have much to do
term3315_pun::~term3315_pun()
{
    // close file if it is open, destroy filename copy
    for(int u=0; u<2; u++) {
        if (TapeMounted(unit_t(u)))
            UnmountTape(unit_t(u));
    }
}


void
term3315_pun::Reset()
{
    Clear();
}


void
term3315_pun::Clear()
{
    m_kbd_busy = false;
    ReaderStop();               // as if XOFF had been seen
}

// ======================================================================
// functions part of term3315_punch_if inheritance.
// just pass them through to the pun subunit.
// ======================================================================

// indiate if a tape is already mounted or not
bool
term3315_pun::TapeMounted(unit_t unit) const
{
    return m_tape_file[unit].is_open();
}


// mount a tape for subsequent reading/writing
void
term3315_pun::MountTape(const unit_t unit, const char *filename)
{
    ASSERT(filename != NULL);               // must be something plausible
    ASSERT(!TapeMounted(unit));

    // first, determine if the file exists
    bool file_exists = false;
    {
        std::ifstream temp_handle;
        temp_handle.open(filename);
        file_exists = temp_handle.is_open();
        temp_handle.close();
    }

    std::fstream::openmode mode;
    if (unit == rdr_unit) {
        // open for reading, at start of file
        mode = std::fstream::in
             | std::fstream::binary;
    } else {
        // open for writing, at end of file
        mode = std::fstream::out
             | std::fstream::in     // (in | out) is required for ::ate to work
             | std::fstream::binary
             | ((file_exists) ? std::fstream::ate
                              : std::fstream::trunc);
    }

    m_tape_file[unit].open(filename, mode);
    if (!m_tape_file[unit].is_open()) {
        UiAlert("Error: can't open file '%s'", filename);
        return;
    }

    m_tape_file[unit].clear();        // clear error bits

    ASSERT(m_tape_filename[unit] == NULL);
    m_tape_filename[unit] = strdup(filename);
    ASSERT(m_tape_filename[unit] != NULL);

    m_ui->PapertapeUpdate(unit, punch_event_t(pe_mount | pe_position));
}


// unmount the currently mounted tape
void
term3315_pun::UnmountTape(const unit_t unit)
{
    ASSERT(TapeMounted(unit));

    m_tape_file[unit].close();

    free(m_tape_filename[unit]);
    m_tape_filename[unit] = NULL;

    if (unit == pun_unit)
        SetPunchEnabled(false);
    else if (unit == rdr_unit)
        ReaderStop();

    m_ui->PapertapeUpdate(unit, pe_mount);
}


char *
term3315_pun::GetTapePath(const unit_t unit) const
{
    return m_tape_filename[unit];
}


// return position of r/w head relative to start, in bytes
// logically this function should be const, but tellg/tellp aren't const.
int
term3315_pun::TapePosition(const unit_t unit)
{
    if (unit == rdr_unit)
        return int(m_tape_file[unit].tellg());  // get pointer
    else
        return int(m_tape_file[unit].tellp());  // put pointer
}


// move r/w position to given point on the tape, in bytes
void
term3315_pun::SetTapePosition(const unit_t unit, int position)
{
    // clear flags, in case EOF was set
    m_tape_file[unit].clear();

    // read and write have separate pointers
    if (unit == rdr_unit)
        m_tape_file[unit].seekg( position, std::fstream::beg );
    else
        m_tape_file[unit].seekp( position, std::fstream::beg );

    // tell gui to have a look at the current position
    m_ui->PapertapeUpdate(unit, pe_position);
}

// ======================================================================
// other hooks
// ======================================================================

// means to associate a keyboard with the punch
void
term3315_pun::BindKeyboard(term3315_kbd *keyboard)
{
    m_kbd = keyboard;
}


// means for punch to receive characters from printer mechanism
void
term3315_pun::ReceivePrtChar(unsigned char byte)
{
    if (TapeMounted(pun_unit) && PunchEnabled()) {
        // raw binary
        // FIXME: to more accurately emulate the behavior of a paper tape punch,
        //        we should do a R/M/W and OR the new byte with the data that
        //        already exists, such that overstrike (most notably RUBOUT)
        //        works.  Of course, if we are at the end of tape, we should
        //        just write the new byte as-is.
        m_tape_file[pun_unit].put(byte);
        m_tape_file[pun_unit].flush();
        m_ui->PapertapeUpdate(pun_unit, pe_position);
    }

    // xon/xoff is tracked only when a tape is loaded
    if (TapeMounted(rdr_unit)) {
        if (byte == 0x11) {             // XON
            ReaderStart();
        } else if (byte == (0x80 | 0x13)) {      // XOFF
	    // the code @44EA in the BASIC interpreter uses this for XOFF,
	    // not 0x13.  I suppose it is because even parity was required.
            ReaderStop(true);
        }
    }
}


// means for punch to indicate it is busy reading a tape
bool
term3315_pun::ReadingTape() const
{
    return TapeMounted(rdr_unit) && ReaderEnabled();
}


// means for keyboard to tell punch that it is ready to accept input
void
term3315_pun::KbdSerialBusy(bool busy)
{
if (g_outf) fprintf(g_outf, "pun: KbdSerialBusy %d\n", busy?1:0);
    m_kbd_busy = busy;
    CheckReader();
}


// if the situation is right, feed a character from the paper tape
// to they keystroke transmission system.
void
term3315_pun::CheckReader()
{
    if (TapeMounted(rdr_unit) && !m_kbd_busy && ReaderEnabled()) {
        // send next character of file
        int ch = m_tape_file[rdr_unit].get();
        if (m_tape_file[rdr_unit].good())
            m_kbd->Keystroke(ch, true);
        else
            ReaderStop();
        m_ui->PapertapeUpdate(rdr_unit, pe_position);

        if (m_tape_xon == ctl_pending)
            ReaderStop();
    }
}


// paper tape reader/punch enable switches
void
term3315_pun::ReaderStart()
{
    m_tape_xon = ctl_xon;
    CheckReader();      // start reading the tape, if appropriate
}

// when the tty receives an xoff, one more character is read before
// the tape reader actually halts.  perhaps it is a bug, but the 3300
// LOAD command, at least, depends on this behavior.  If xoff takes
// immediate effect, something goes wrong in the interrupt handling logic
// such that XON never gets sent, and the tape just hangs at the end of
// each line (which is when XOFF is sent).  By sending one more character,
// the interrupt routine gets kick started again.
void
term3315_pun::ReaderStop(bool inertia)
{
    m_tape_xon = (inertia) ? ctl_pending : ctl_xoff;
}


bool
term3315_pun::ReaderEnabled() const
{
    return (m_tape_xon != ctl_xoff);
}

// reader/punch enable switches
void
term3315_pun::SetPunchEnabled(bool enable)
{
    if (enable != m_tape_punch_enabled) {
        m_tape_punch_enabled = enable;
        m_ui->PapertapeUpdate(pun_unit, pe_punchen);
    }
}

bool
term3315_pun::PunchEnabled() const
{
    return m_tape_punch_enabled;
}
