// this code emulate a Wang 3320 terminal controller connected to a
// Wang 3315 terminal, which is a rebadged ASR33 TTY.
//
// This specific file emulates just the printer output part of the TTY.

#include "system3300.h"
#include "term3315.h"
#include "term3315_impl.h"
#include "scheduler.h"
#include "cpu3300.h"
#include "UiTerm3315.h"
#include "UiSystem.h"

extern FILE *g_outf;


term3315_prt::term3315_prt(cpu3300 *cpu, Scheduler &scheduler,
                           term3315_pun *punch,
                           int termnum, int station, UiTerm3315 *ui) :
    m_ui(ui),
    m_cpu(cpu),
    m_pun(punch),
    m_scheduler(scheduler),
    m_termid(termnum),
    m_prt_ioaddr(   0x21 + 2*termnum),
    m_prt_memaddr(0x0221 + 2*termnum)
{
    ASSERT(termnum >= 0 && termnum < 16);

    m_prt_status = 0x00;  // cleared
    m_prt_timer  = NULL;  // 

    // milliseconds per bit of serial transfer
    m_prt_bps = 110;  // 110 baud  FIXME: make this configurable

    // link terminal receive to cpu
    m_cpu->AddIoDevice(this, station, m_prt_ioaddr);
}


// destructor doesn't have much to do
term3315_prt::~term3315_prt()
{
    Clear(); // kills off timer

    // remove this device from the interrupt chain
    m_cpu->RemoveIoDevice(m_prt_ioaddr);
}


void
term3315_prt::Reset()
{
    Clear();
}


void
term3315_prt::Clear()
{
    ENSURE_TIMER_DEAD(m_prt_timer);
    m_prt_status = 0x00;  // cleared
    m_cpu->IoInterrupt(m_prt_ioaddr, false);
}


// CIOJ behavior
void
term3315_prt::WrControl(int addr, int data)
{
    ASSERT(m_prt_ioaddr == addr);
    const int t_bit = TIMER_US(1000000.0 / m_prt_bps);

    if (m_prt_status >= 0x10 && m_prt_status < 0xC0) {
        // control words aren't allowed during transfer
        m_prt_status = 0xFF;
        return;
    }

    ENSURE_TIMER_DEAD(m_prt_timer);
    // p.28 of the 3300 Reference Manual:
    //     "A pending output interrupt will never be cancelled
    //      unless status is reset by a CIOJ."
if (g_outf) fprintf(g_outf, "prt: about to remove interrupt due to WrControl\n");
    m_cpu->IoInterrupt(m_prt_ioaddr, false); // clear interrupt

    if (data == 0x83) {
        // clear status command
        m_prt_status = 0x00;
    } else if (data == 0x08) {
        // clear status command and write data
        m_prt_status = 0x10;
        // kick off timer based sequencer
        m_prt_timer = m_scheduler.TimerCreate(t_bit, *this, &term3315_prt::prtcb, 1);
    } else {
        UiAlert("Got unexpected control byte 0x%02X at printer device", data);
        m_prt_status = 0x00;
    }
}


// this gets called back after the serial port delay to communicate the
// character has been transferred.  Page 28 of the "3300 Computer Reference
// Manual" is contradictory on exactly the handshake timing occurs
// (says status proceeds from 0x10 .. 0xB0, then the character is ready and
//  status goes to 0xC0, yet the first 10 states occur in cycles "1 thru 10".
//
// Further, page 29, section C.3 says:
//    an interrupt is issued after the transfer cycles are completed
//    at the beginning of cycle 11 (90.91 ms after the start of write).
//    The ready condition and/or interrupt may be processed any time after
//    cycle 11 and is not lost.
// This would mean that the interrupt is raised when status changes from
// 0xA0 to 0xB0 (the start of the 11th cycle).  But all the other text is
// geared around nothing happening until status is 0xC0.  How could this
// work to raise the interrupt at 90.91 ms, as presumably the TTY can not
// operate faster than one char every 11*9.09 ms, or 100 ms.
void
term3315_prt::prtcb(int k)
{
    const int t_bit = TIMER_US(1000000.0 / m_prt_bps);

    if (k < 11) {
        // count 0x10, 0x20, 0x30, ... 0xB0
        m_prt_status = (k << 4) + 0x10;
        m_prt_timer = m_scheduler.TimerCreate(t_bit, *this, &term3315_prt::prtcb, k+1);
    } else {
        // reached terminal count -- transfer the character
        // DMA fetch the character from the memory
        // p. 30 of the Reference Manual says the memory location is left 0x00
        unsigned char byte = m_cpu->GetMem8(m_prt_memaddr);
        m_cpu->SetMemory(m_prt_memaddr, 0x00);
        m_ui->OutChar((uint8)byte);

        // send the character to the punch for consideration
        m_pun->ReceivePrtChar(byte);

        m_prt_status = 0xC0;
        m_prt_timer  = NULL;
if (g_outf) fprintf(g_outf, "prt: about to interrupt\n");
        m_cpu->IoInterrupt(m_prt_ioaddr, true);
    }
}


int
term3315_prt::RdStatus(int addr) const
{
    ASSERT(m_prt_ioaddr == addr);
    return m_prt_status;
}

void
term3315_prt::WrData(int addr, int data)
{
    ASSERT(m_prt_ioaddr == addr);
    ASSERT(0);
}

int
term3315_prt::RdData(int addr) const
{
    ASSERT(m_prt_ioaddr == addr);
    ASSERT(0);
    return 0xFF;   // just in case somebody goes and does it
}

// AKIJ behavior
void
term3315_prt::IAck(int addr)
{
    ASSERT(m_prt_ioaddr == addr);

    // p. 28 of the Reference Manual says status is cleared only in READY state
    if (m_prt_status == 0xC0) {
        m_prt_status = 0x00;
    }

#if 1
    // p.28 of the 3300 Reference Manual:
    //     "A pending output interrupt will never be cancelled
    //      unless status is reset by a CIOJ."
    // HOWEVER, empirically, the BASIC interpreter stays stuck in
    //          an ISR forever loop, as the raised interrrupt is
    //          not followed by any CIOJ command if there is no
    //          other output pending.
if (g_outf) fprintf(g_outf, "prt: about to remove interrupt due to IAck\n");
    m_cpu->IoInterrupt(m_prt_ioaddr, false); // clear interrupt
#endif
}
