// this code emulate a Wang 3320 terminal controller connected to a
// Wang 3315 terminal, which is a rebadged ASR33 TTY.
//
// This specific file emulates just the keyboard input part of the TTY.

#include "system3300.h"
#include "term3315.h"
#include "term3315_impl.h"
#include "scheduler.h"
#include "cpu3300.h"
#include "UiSystem.h"

extern FILE *g_outf;


term3315_kbd::term3315_kbd(cpu3300 *cpu, Scheduler &scheduler,
                           term3315_pun *punch, int termnum, int station) :
    m_cpu(cpu),
    m_pun(punch),
    m_scheduler(scheduler),
    m_termid(termnum),
    m_kbd_ioaddr(   0x20 + 2*termnum),
    m_kbd_memaddr(0x0220 + 2*termnum)
{
    ASSERT(termnum >= 0 && termnum < 16);

    m_kbd_status = 0x00;  // cleared
    m_kbd_timer  = NULL;

    // milliseconds per bit of serial transfer
    m_kbd_bps = 110;  // 110 baud  FIXME: make this configurable

    // link terminal transmit to cpu
    m_cpu->AddIoDevice(this, station, m_kbd_ioaddr);
}


// destructor doesn't have much to do
term3315_kbd::~term3315_kbd()
{
    Clear(); // kills off timer

    // remove this device from the interrupt chain
    m_cpu->RemoveIoDevice(m_kbd_ioaddr);
}


// mechanism for UI to notify us of keystroke to our terminal
// fromTape is true if the character came from the paper tape unit
void
term3315_kbd::Keystroke(uint8 keycode, bool fromTape)
{
    const int t_bit = TIMER_US(1000000.0 / m_kbd_bps);

if (g_outf) fprintf(g_outf, "kbd: Keystroke(0x%02X), %d\n", keycode, fromTape);
    // ignore any keystroke that appears before previous one is done
    if (m_kbd_status != 0x00 && m_kbd_status != 0xFF)
        return;

    // ignore any keyboard keystroke while the paper tape is reading
    if (!fromTape && m_pun->ReadingTape())
        return;

    m_kbd_key    = keycode;
    m_kbd_status = 0x10;

    // kick off timer based sequencer
    ASSERT(m_kbd_timer == NULL);
if (g_outf) fprintf(g_outf, "kbd: about to create a timer 1\n");
    m_kbd_timer = m_scheduler.TimerCreate(t_bit, *this, &term3315_kbd::keycb, 1);
    m_pun->KbdSerialBusy(true);
}


// this gets called back after the serial port delay to communicate the
// character has been transferred.  Page 28 of the "3300 Computer Reference
// Manual" is contradictory on exactly the handshake timing occurs
// (says status proceeds from 0x10 .. 0xA0, then the character is ready and
//  status goes to 0xB0, yet the first 10 states occur in cycles "1 thru 9".
void
term3315_kbd::keycb(int k)
{
    const int t_bit = TIMER_US(1000000.0 / m_kbd_bps);
if (g_outf) fprintf(g_outf, "kbd:keycb, status=0x%02X\n", m_kbd_status);

    if (k < 10) {
        // count 0x10, 0x20, 0x30, ... 0x90
        m_kbd_status = (k << 4) + 0x10;
if (g_outf) fprintf(g_outf, "kbd: about to create a timer 2\n");
        m_kbd_timer = m_scheduler.TimerCreate(t_bit, *this, &term3315_kbd::keycb, k+1);
    } else if (k == 10) {
        // reached terminal count -- transfer the character
        m_kbd_status = 0xB0;
        m_cpu->SetMemory(m_kbd_memaddr, m_kbd_key);
if (g_outf) fprintf(g_outf, "kbd: about to interrupt\n");
        m_cpu->IoInterrupt(m_kbd_ioaddr, true);
        // page 29, section C.1.d, of the reference manual has this to say:
        //    The input status is reset back to 00, (idle), 13.5 milliseconds
        //    after data transfer is complted (B0 ready status) if not reset
        //    by an AKI or CIOJ.
        // at 110 baud, each bit is 9.09 ms, so 13.5 ms is 1.5 bit periods.
if (g_outf) fprintf(g_outf, "kbd: about to create a timer 3\n");
        m_kbd_timer = m_scheduler.TimerCreate(1.5*t_bit, *this, &term3315_kbd::keycb, 11);
    } else {
        // page 28 of the 3300 Reference Manual, footnote (2) says that
        // the pending interrupt will be cancelled and the status returned
        // to 00 if software hasn't done it itself with either CIOJ or AKIJ.
        m_kbd_timer = NULL;
        // clear the interrupt request if it wasn't already handled by AKIJ or CIOJ
        if (m_kbd_status != 0x00) {
            m_kbd_status = 0x00;
if (g_outf) fprintf(g_outf, "kbd: about to remove interrupt due to timeout\n");
            m_cpu->IoInterrupt(m_kbd_ioaddr, false);
        }
        // tell the paper tape reader that the previous character has been
        // transmitted, giving it the chance to queue up another one
        m_pun->KbdSerialBusy(false);
    }
}


void
term3315_kbd::Reset()
{
    Clear();
}


void
term3315_kbd::Clear()
{
    ENSURE_TIMER_DEAD(m_kbd_timer);
    m_kbd_status = 0x00;  // cleared
if (g_outf) fprintf(g_outf, "kbd: about to remove interrupt due to Clear\n");
    m_cpu->IoInterrupt(m_kbd_ioaddr, false);
}


// CIOJ behavior
void
term3315_kbd::WrControl(int addr, int data)
{
if (g_outf) fprintf(g_outf, "kbd: got CIOJ %02X, %02X\n", addr, data);
    ASSERT(m_kbd_ioaddr == addr);

    if (m_kbd_status >= 0x10 && m_kbd_status < 0xB0) {
        // p28 of Reference manual:
        // control words aren't allowed during transfer
        m_kbd_status = 0xFF;
    } else if (data != 0x83) {
        UiAlert("Got unexpected control byte 0x%02X at keyboard device", data);
        m_kbd_status = 0x00;
    } else {

        ENSURE_TIMER_DEAD(m_kbd_timer);
        m_kbd_status = 0x00;
        m_pun->KbdSerialBusy(false);  // after xmission, let pt reader know

        // p28 of Reference manual: status is cleared when CIOJ w/A=0x83
        if (0) { // FIXME: resolve this -- clear it or not?
            // a careful reading of the reference manual never indicates that
            // a pending interrupt is cleared by a CIOJ 0x83 operation.
if (g_outf) fprintf(g_outf, "kbd: about to remove interrupt due to CIOJ %02X, %02X\n", addr, data);
            m_cpu->IoInterrupt(m_kbd_ioaddr, false);
        }
    }
}


int
term3315_kbd::RdStatus(int addr) const
{
    ASSERT(m_kbd_ioaddr == addr);
    return m_kbd_status;
}


void
term3315_kbd::WrData(int addr, int data)
{
    ASSERT(m_kbd_ioaddr == addr);
    ASSERT(0);
}


int
term3315_kbd::RdData(int addr) const
{
    ASSERT(m_kbd_ioaddr == addr);
    ASSERT(0);
    return 0xFF;       // just in case somebody goes and does it
}


// AKIJ behavior
void
term3315_kbd::IAck(int addr)
{
    ASSERT(m_kbd_ioaddr == addr);

    // p28 of Reference manual: status is cleared when AKIJ command
    // is executed during cycles 10-11 of the transfer
    if ((m_kbd_status & 0xF0) == 0xB0) {
        int old_kbd_status = m_kbd_status;
        m_kbd_status = 0x00;
if (g_outf) fprintf(g_outf, "kbd: about to remove interrupt due to IAck %02X\n", addr);
        m_cpu->IoInterrupt(m_kbd_ioaddr, false);
    }

    // we intentionally don't notify the paper tape reader that we
    // are ready for another character here.  if we did, it could cause
    // a call to Keystroke(), which would change status back to 0x10,
    // the the CIOJ that often follows an interrupt would want to
    // clear the state to 0x00, interrupting the progress of the char
    // just queued.  instead we let CIOJ notify the paper tape punch,
    // or if that doesn't happen, the keycb timeout.
}
