// emulate a Wang 3315 terminal, which is a rebadged ASR33 TTY.
//

#ifndef _TERM3315_IMPL_H_
#define _TERM3315_IMPL_H_

#include "IoDevice.h"
#include "term3315.h"
#include <fstream>    // not needed now that term3315_pun isn't here

// forward class references
class Timer;

// ======================================================================
// keyboard subcomponent
// ======================================================================

class term3315_kbd: public IoDevice
{
public:
    // create terminal #"termnum" and attach it to the CPU
    term3315_kbd(cpu3300 *cpu, Scheduler &scheduler, term3315_pun *punch,
                 int termnum, int station);

    // clean up
    ~term3315_kbd();

    // mechanism for UI to notify us of keystroke to our terminal
    // fromTape is true if the character came from the paper tape unit
    void Keystroke(uint8 keycode, bool fromTape=false);

    // overridden IoDevice methods
    void Reset();
    void Clear();
    void WrControl(int addr, int data);
    int RdStatus(int addr) const;
    void WrData(int addr, int data);
    int RdData(int addr) const;
    void IAck(int addr);

private:
    // keyboard callback
    void keycb(int k);

    cpu3300 * const m_cpu;      // cache the cpu we are attached to
    const int m_termid;         // terminal number

    Scheduler    &m_scheduler;  // scheduling agent
    term3315_pun *m_pun;        // associated punch unit

    // keyboard state
    const uint8  m_kbd_ioaddr;          // i/o address of this device
    const uint16 m_kbd_memaddr;         // memory address of character buffer
          uint8  m_kbd_status;          // status to send to CPU
          Timer *m_kbd_timer;           // timer handle
          uint8  m_kbd_key;             // pending keystroke
          int    m_kbd_bps;             // keyboard baud rate
};


// ======================================================================
// printer subcomponent
// ======================================================================

class term3315_prt: public IoDevice
{
public:
    // create terminal #"termnum" and attach it to the CPU
    term3315_prt(cpu3300 *cpu, Scheduler &scheduler, term3315_pun *punch,
                 int termnum, int station, UiTerm3315 *ui);

    // clean up
    ~term3315_prt();

    // overridden IoDevice methods
    void Reset();
    void Clear();
    void WrControl(int addr, int data);
    int RdStatus(int addr) const;
    void WrData(int addr, int data);
    int RdData(int addr) const;
    void IAck(int addr);

private:
    // printer callback
    void prtcb(int k);

    UiTerm3315 * const m_ui;    // connection to user interface
    cpu3300    * const m_cpu;   // cache the cpu we are attached to
    const int m_termid;         // terminal number

    Scheduler    &m_scheduler;  // scheduling agent
    term3315_pun *m_pun;        // associated punch unit

    // printer state
    const uint8  m_prt_ioaddr;  // i/o address of this device
    const uint16 m_prt_memaddr; // memory address of character buffer
          uint8  m_prt_status;  // status to send to CPU
          Timer *m_prt_timer;   // schedular handle
          int    m_prt_bps;     // printer baud rate
};


// ======================================================================
// papertape reader/punch subcomponent
// ======================================================================

// The punch isn't an IoDevice; it instead is notified by the prt submodule
// when a character is received, and it may send characters to the kbd
// submodule via the kbd Keystroke() function.
class term3315_pun
{
public:
    term3315_pun(int termnum, UiTerm3315 *ui);
    ~term3315_pun();

    // control methods
    void Reset();
    void Clear();

    // means to associate a keyboard with the punch
    void BindKeyboard(term3315_kbd *keyboard);

    // means for keyboard to tell punch that it is ready to accept input
    void KbdSerialBusy(bool busy);

    // means for punch to receive characters from printer mechanism
    void ReceivePrtChar(unsigned char byte);

    // paper tape file attributes
    bool  TapeMounted(const unit_t unit) const;
    void  MountTape(const unit_t unit, const char *filename);
    void  UnmountTape(const unit_t unit);
    char *GetTapePath(const unit_t unit) const;
    int   TapePosition(const unit_t unit);
    void  SetTapePosition(const unit_t unit, int position);

    // means for punch to indicate it is busy reading a tape
    bool ReadingTape() const;

    // paper tape reader/punch enable switches
    void ReaderStart();
    void ReaderStop(bool inertia = false);
    bool ReaderEnabled() const;
    bool PunchEnabled() const;
    void SetPunchEnabled(bool enable);

private:
    const int m_termid;         // terminal number
    UiTerm3315 * const m_ui;    // connection to user interface
    term3315_kbd *m_kbd;        // associated keyboard unit
    bool m_kbd_busy;            // most recent notice from KbdSerialBusy()

    // if the situation is right, feed a character from the paper tape
    // to they keystroke transmission system.
    void CheckReader();

    std::fstream m_tape_file[2];    // file handle
    char *m_tape_filename[2];       // file representing paper tape
    enum { ctl_xoff,        // not reading tape
           ctl_pending,     // an xoff is pending (due to inertia)
           ctl_xon }        // reading tape
          m_tape_xon;               // tracks reader XON/XOFF
    bool  m_tape_punch_enabled;     // paper tape punch control
};

#endif // _TERM3315_IMPL_H_
