// this code emulate a Wang 3320 terminal controller connected to a
// Wang 3315 terminal, which is a rebadged ASR33 TTY.
//
// Mostly this just passes requests through to the appropriate subunit.

#include "system3300.h"
#include "term3315.h"
#include "term3315_impl.h"
#include "UiTerm3315.h"


// create a new terminal emulation core
term3315::term3315(cpu3300 *cpu, Scheduler &scheduler, int termnum) :
    m_ui(0),
    m_kbd(0),
    m_prt(0),
    m_pun(0),
    m_initialized(false)
{
    ASSERT(termnum >= 0 && termnum < 16);

    const int station = (termnum >> 2); // by convention

    // create components
    m_ui  = new UiTerm3315(termnum, this);
    m_pun = new term3315_pun(termnum, m_ui);
    m_kbd = new term3315_kbd(cpu, scheduler, m_pun, termnum, station);
    m_prt = new term3315_prt(cpu, scheduler, m_pun, termnum, station, m_ui);

    // tell m_prt which keyboard it is associated with
    m_pun->BindKeyboard(m_kbd);

    m_initialized = true;
}


term3315::~term3315()
{
    delete m_prt;
    delete m_kbd;
    delete m_pun;
    m_ui->Destroy();    // it is a wxWindow, it must be Destroyed
}


void
term3315::Reset()
{
    ASSERT(Initialized());
    m_kbd->Reset();
    m_prt->Reset();
    m_pun->Reset();
}

void
term3315::Clear()
{
    ASSERT(Initialized());
    m_kbd->Clear();
    m_prt->Clear();
    m_pun->Clear();
}

void
term3315::WrControl(int addr, int data)
{
    ASSERT(Initialized());
    IoDevice *subunit = (addr & 1) ? static_cast<IoDevice*>(m_prt)
                                   : static_cast<IoDevice*>(m_kbd);
    subunit->WrControl(addr, data);
}

int
term3315::RdStatus(int addr) const
{
    ASSERT(Initialized());
    IoDevice *subunit = (addr & 1) ? static_cast<IoDevice*>(m_prt)
                                   : static_cast<IoDevice*>(m_kbd);
    return subunit->RdStatus(addr);
}

void
term3315::WrData(int addr, int data)
{
    ASSERT(Initialized());
    IoDevice *subunit = (addr & 1) ? static_cast<IoDevice*>(m_prt)
                                   : static_cast<IoDevice*>(m_kbd);
    subunit->WrData(addr, data);
}

int
term3315::RdData(int addr) const
{
    ASSERT(Initialized());
    IoDevice *subunit = (addr & 1) ? static_cast<IoDevice*>(m_prt)
                                   : static_cast<IoDevice*>(m_kbd);
    return subunit->RdData(addr);
}

void
term3315::IAck(int addr)
{
    ASSERT(Initialized());
    IoDevice *subunit = (addr & 1) ? static_cast<IoDevice*>(m_prt)
                                   : static_cast<IoDevice*>(m_kbd);
    subunit->IAck(addr);
}

// ======================================================================
// functions not part of IoDevice inheritance
// ======================================================================

// mechanism for UI to notify us of keystroke to our terminal
void
term3315::Keystroke(uint8 keycode)
{
    if (!Initialized()) return;
    m_kbd->Keystroke(keycode);
}

// ======================================================================
// proxy down to the pun subunit
// ======================================================================

void
term3315::ReaderStart()
{
    ASSERT(Initialized());
    m_pun->ReaderStart();
}

void
term3315::ReaderStop()
{
    ASSERT(Initialized());
    m_pun->ReaderStop();
}

bool
term3315::ReaderEnabled() const
{
    if (!Initialized()) return false;
    return m_pun->ReaderEnabled();
}

void
term3315::SetPunchEnabled(bool enable)
{
    ASSERT(Initialized());
    m_pun->SetPunchEnabled(enable);
}

bool
term3315::PunchEnabled() const
{
    if (!Initialized()) return false;
    return m_pun->PunchEnabled();
}

// indiate if a tape is already mounted or not
bool
term3315::TapeMounted(const unit_t unit) const
{
    if (!Initialized()) return false;
    return m_pun->TapeMounted(unit);
}


// mount a tape for subsequent reading/writing
void
term3315::MountTape(const unit_t unit, const char *filename)
{
    ASSERT(Initialized());
    m_pun->MountTape(unit, filename);
}

// unmount the currently mounted tape
void
term3315::UnmountTape(const unit_t unit)
{
    ASSERT(Initialized());
    m_pun->UnmountTape(unit);
}


char *
term3315::GetTapePath(const unit_t unit) const
{
    if (!Initialized()) return "";
    return m_pun->GetTapePath(unit);
}

// return position of r/w head relative to start, in bytes
int
term3315::TapePosition(const unit_t unit)
{
    if (!Initialized()) return 0;
    return m_pun->TapePosition(unit);
}


// move r/w position to given point on the tape, in bytes
void
term3315::SetTapePosition(const unit_t unit, int position)
{
    ASSERT(Initialized());
    m_pun->SetTapePosition(unit, position);
}
