// emulate a Wang 3320 cassette tape interface, from the Wang 3300 system.

#ifndef _TERM3320_H_
#define _TERM3320_H_

#include "IoDevice.h"

// forward class references
class cpu3300;
class Scheduler;
class UiTape3320;

class tape3320: public IoDevice
{
public:
    // create terminal #"termnum" and attach it to the CPU
    tape3320(cpu3300 *cpu, Scheduler &scheduler, int unit);

    // clean up
    ~tape3320();

    // overridden IoDevice methods
    void Reset();
    void Clear();
    void WrControl(int addr, int data);
    int  RdStatus(int addr) const;
    void WrData(int addr, int data);
    int  RdData(int addr) const;
    void IAck(int addr);

    // cassette tape file attributes
    bool  TapeMounted(const unit_t unit) const;
    void  MountTape(const unit_t unit, const char *filename);
    void  UnmountTape(const unit_t unit);
    char *GetTapePath(const unit_t unit) const;
    int   TapePosition(const unit_t unit);
    void  SetTapePosition(const unit_t unit, int position);

private:
    // scheduled state change callback function
    void timer_state_cb(int k);
    // scheduled data transfer callback function
    void timer_byte_cb(int k);

    UiTape3320   *m_ui;             // user interface routines
    int           m_unit;	    // which tape unit this is

    // status bits; some bits are common for the read and write
    // addresses, others are common.  see the 3320 programming guide.
    bool          m_rstat0;     // 1=not ready with input data
    bool          m_wstat0;     // 1=not ready to accept command/data
    bool          m_rstat1;     // 1=input error
    bool          m_wstat1;     // 1=output error
    bool          m_stat2;      // 1=end of tape (EOT) sensed
    bool          m_stat3;      // 1=file protect switch set
    bool          m_stat4;      // 1=end of block detected (no flux transitions)
    bool          m_stat5;      // 1=read mode on
    bool          m_stat6;      // 1=write mode on
    bool          m_stat7;      // 1=beginning of tape (BOT) sensed

    // There are a few different states the tape machine can be in, and
    // different events cause transitions between them, often based on time.
    // states:
    enum {
        empty,      // no  tape in drive
        idle,       // tape in drive, but not moving
        rew,        // high speed tape motion in reverse direction
        ffwd,       // high speed tape motion in forward direction
        read0,      // normal speed forward tape motion; read not yet enabled.
                    // it transitions to the read state after 30 ms.
        read,       // normal speed forward tape motion, actively reading
        write0,     // normal speed forward tape motion, erase is enabled, write
                    // not yet enabled.  transitions to the write state after 50 ms.
        write,      // normal speed forward tape motion, actively writing
        halt,       // interval while tape stops motion.
                    // it transitions to idle at the end of a 40 ms timer.
        gap0,       // enables erase mode for 40 ms, then raises an interrupt
                    // (if enabled) and transitions to the gap state.
        gap,        // erase mode is enabled and the tape is left in motion
                    // until a new command arrives.
        fspace0,    // normal speed forward motion until the next gap is
                    // detected, then interrupt (if enabled) is raised, and
                    // transitions to the fspace state.
        fspace,     // the tape remains in normal forward motion until a
                    // new command says otherwise (or EOT I suppose)
        bspace0,    // normal speed reverse motion until the next gap is
                    // detected, then interrupt (if enabled) is flagged and
                    // transitions to the bspace state.
        bspace      // the tape remains in normal reverse motion until a
                    // new command says otherwise (or BOT I suppose)
    } m_state;

    Timer        *m_state_timer;    // timer handle
    Timer        *m_byte_timer;     // timer handle
};

#endif // _TERM3320_H_
