// This code emulates a Wang 3320 cassette tape subsystem.

#include "system3300.h"
#include "tape3320.h"
#include "UiTape3320.h"


// create a new terminal emulation core
tape3320::tape3320(cpu3300 *cpu, Scheduler &scheduler, int unit) :
    m_ui(0),
    m_unit(unit),
    m_rstat0(false), m_wstat0(false),
    m_rstat1(false), m_wstat1(false),
    m_stat2(false), m_stat3(false),
    m_stat4(false), m_stat5(false),
    m_stat6(false), m_stat7(false),
    m_state_timer(NULL),
    m_byte_timer(NULL)
{
    ASSERT(unit >= 0 && unit < 4);

    // connect to UI
    m_ui = new UiTape3320(unit, this);
}


tape3320::~tape3320()
{
    Clear(); // kills off timer

    // remove this device from the interrupt chain
    m_cpu->RemoveIoDevice(m_ioaddr);

    m_ui->Destroy();    // it is a wxWindow, it must be Destroyed, not deleted
}


void
tape3320::Reset()
{
    Clear();
}


void
tape3320::Clear()
{
    ENSURE_TIMER_DEAD(m_timer);
    m_status = 0x00;  // cleared
    m_cpu->IoInterrupt(m_ioaddr, false);

    m_rstat0 = true;    // read data byte not available
    m_wstat0 = true;    // not ready to accept write data
    m_rstat1 = false;   // no input error
    m_wstat1 = false;   // no output error
//  m_stat2  = false;   // FIXME: EOT determined by tape state
//  m_stat3  = false;   // FIXME: write prot determined by GUI interaction
    m_stat4  = true;    // end of block (probably doesn't matter)
    m_stat5  = false;   // read mode off
    m_stat6  = false;   // write mode off
//  m_stat7  = false;   // FIXME: BOT determined by tape state
}


// There are a few different states the tape machine can be in, and
// different events cause transitions between them, often based on time.
// see the definition of the m_state in tape3320.h for the meaning of each
// state.
//
// FIXME: What if we we see EOT before read0 times out?
//        What if we we see EOT before gap0 times out?
//        What if we we see EOT before fspace0 times out?
//        What if we we see BOT before bspace times out?

void
tape3320::timer_byte_cb(int k)
{
    // number of microseconds per byte transferred
    const float byte_us = TIMER_US(151*8);  // 151 us/bit, roughly

    // FIXME:

    m_prt_status = ((k+1) << 4);
    m_byte_timer = m_scheduler.TimerCreate(t_bit, *this, &term3315_prt::prtcb, k+1);

    m_cpu->SetMemory(m_prt_memaddr, 0x00);

    m_cpu->IoInterrupt(m_prt_ioaddr, true);
}


// CIOJ behavior
void
tape3320::WrControl(int addr, int data)
{
    ASSERT(m_prt_ioaddr == addr);
    const int t_bit = TIMER_US(1000000.0 / m_prt_bps);

    ENSURE_TIMER_DEAD(m_byte_timer);
    // p.28 of the 3300 Reference Manual:
    //     "A pending output interrupt will never be cancelled
    //      unless status is reset by a CIOJ."
if (g_outf)
fprintf(g_outf, "prt: about to remove interrupt due to WrControl\n");
    m_cpu->IoInterrupt(m_prt_ioaddr, false); // clear interrupt

    if (m_prt_status >= 0x10 && m_prt_status < 0xC0) {
        // control words aren't allowed during transfer
        m_prt_status = 0xFF;
    } else if (data == 0x83) {
        // clear status command
        m_prt_status = 0x00;
    } else if (data == 0x08) {
        // clear status command and write data
        m_prt_status = 0x10;
        // kick off timer based sequencer
        m_byte_timer = m_scheduler.TimerCreate(t_bit, *this, &term3315_prt::prtcb, 1);
    } else {
        // FIXME: UiAlert("Got unexpected control byte 0x%02X at printer device", *data);
        fprintf(stderr, "Got unexpected control byte 0x%02X at printer device", data);
        m_prt_status = 0x00;
    }
}


int
tape3320::RdStatus(int addr) const
{
    if (addr & 1) {
        // write port status
        return (((m_wstat0) ? 1 : 0) << 0) |
               (((m_wstat1) ? 1 : 0) << 1) |
               (((m_stat2)  ? 1 : 0) << 2) |
               (((m_stat3)  ? 1 : 0) << 3) |
               (((m_stat4)  ? 1 : 0) << 4) |
               (((m_stat5)  ? 1 : 0) << 5) |
               (((m_stat6)  ? 1 : 0) << 6) |
               (((m_stat7)  ? 1 : 0) << 7) ;
    } else {
        // read port status
        return (((m_rstat0) ? 1 : 0) << 0) |
               (((m_rstat1) ? 1 : 0) << 1) |
               (((m_stat2)  ? 1 : 0) << 2) |
               (((m_stat3)  ? 1 : 0) << 3) |
               (((m_stat4)  ? 1 : 0) << 4) |
               (((m_stat5)  ? 1 : 0) << 5) |
               (((m_stat6)  ? 1 : 0) << 6) |
               (((m_stat7)  ? 1 : 0) << 7) ;
    }
    ASSERT(m_ioaddr == addr);
    return m_status;
}


void
tape3320::WrData(int addr, int data)
{
    ASSERT(m_prt_ioaddr == addr);
    ASSERT(0);
    // TBD
}


int
tape3320::RdData(int addr) const
{
    ASSERT(m_prt_ioaddr == addr);
    // TBD
    ASSERT(0);
    return 0xFF;   // just in case somebody goes and does it
}


// AKIJ behavior
void
tape3320::IAck(int addr)
{
    ASSERT(m_prt_ioaddr == addr);

    // p. 28 of the Reference Manual says status is cleared only in READY state
    if (m_prt_status == 0xC0) {
        m_prt_status = 0x00;
    }

#if 1
    // p.28 of the 3300 Reference Manual:
    //     "A pending output interrupt will never be cancelled
    //      unless status is reset by a CIOJ."
    // HOWEVER, empirically, the BASIC interpreter stays stuck in
    //          an ISR forever loop, as the raised interrrupt is
    //          not followed by any CIOJ command if there is no
    //          other output pending.
if (g_outf)
fprintf(g_outf, "prt: about to remove interrupt due to IAck\n");
    m_cpu->IoInterrupt(m_prt_ioaddr, false); // clear interrupt
#endif
    // TBD
}

// ======================================================================
// 3320-specific methods
// ======================================================================

// indiate if a tape is already mounted or not
bool
tape3320::TapeMounted(const unit_t unit) const
{
    return m_tape_file[unit].is_open();
}


// mount a tape for subsequent reading/writing
void
tape3320::MountTape(const unit_t unit, const char *filename)
{
    ASSERT(filename != NULL);               // must be something plausible
    ASSERT(!TapeMounted(unit));

    std::fstream::openmode mode = std::fstream::out
                                | std::fstream::in
                                | std::fstream::binary;

    m_tape_file[unit].open(filename, mode);
    if (!m_tape_file[unit].is_open()) {
        UiAlert("Error: can't open file '%s'", filename);
        return;
    }

    m_tape_file[unit].clear();        // clear error bits

    ASSERT(m_tape_filename[unit] == NULL);
    m_tape_filename[unit] = strdup(filename);
    ASSERT(m_tape_filename[unit] != NULL);

//  m_ui->PapertapeUpdate(unit, punch_event_t(pe_mount | pe_position));
}

// unmount the currently mounted tape
void
tape3320::UnmountTape(const unit_t unit)
{
    ASSERT(TapeMounted(unit));

    m_tape_file[unit].close();

    free(m_tape_filename[unit]);
    m_tape_filename[unit] = NULL;

//  m_ui->PapertapeUpdate(unit, pe_mount);
}


char *
tape3320::GetTapePath(const unit_t unit) const
{
    return m_tape_filename[unit];
}

// return position of r/w head relative to start, in bytes
int
tape3320::TapePosition(const unit_t unit)
{
// FIXME: which one?
    return int(m_tape_file[unit].tellg());  // get pointer
//  return int(m_tape_file[unit].tellp());  // put pointer
}


// move r/w position to given point on the tape, in bytes
void
tape3320::SetTapePosition(const unit_t unit, int position)
{
    // clear flags, in case EOF was set
    m_tape_file[unit].clear();

    // read and write have separate pointers
// FIXME: which one?
    m_tape_file[unit].seekg( position, std::fstream::beg );
    m_tape_file[unit].seekp( position, std::fstream::beg );

    // tell gui to have a look at the current position
//  m_ui->PapertapeUpdate(unit, pe_position);
}
