// ======================================================================
// System3300 contains the various components of the system:
//    cpu
//    terminal(s)
//    cassette drive
//    scheduler (this is a logical construct, of course)
// ======================================================================

#ifndef _SYSTEM3300_H_
#define _SYSTEM3300_H_

#include "w3300.h"      // basic types

// forward class references
class cpu3300;
class term3315;
class Scheduler;

#include <vector>
using std::vector;

// =============================================================
// system configuration class
// =============================================================
// The idea is to encapsulate some of the gross system configuration
// state into a central place.  the main benefit is that when changing
// configuration, there are two sets of configuration: the current one
// and the one being twiddled.  this makes it easy to do.

class SystemConfig
{
public:
    SystemConfig();                         // gets config from .ini file
    SystemConfig(const SystemConfig &obj);  // copy constructor
    SystemConfig &operator=(const SystemConfig &rhs);  // assignment
    ~SystemConfig();

    void Load();          // read from configuration file
    void Save() const;    // save to configuration file

    bool GetCpuSpeedRegulated() const { return m_cpu_speed_regulated; }
    int  GetRamKB()             const { return m_ram_kb; }
    int  GetNumTerminals()      const { return m_num_terminals; }
    int  GetNumCassetteDrives() const { return m_num_cassette_drives; }

    void SetCpuSpeedRegulated(bool v);
    void SetRamKB(int v);
    void SetNumTerminals(int v);
    void SetNumCassetteDrives(int v);

private:
    bool m_cpu_speed_regulated;
    int  m_ram_kb;
    int  m_num_terminals;
    int  m_num_cassette_drives;
};

// =============================================================
// system class
// =============================================================

class System3300
{
public:
     System3300();   // constructor
    ~System3300();   // destructor

    // give access to components
    static Scheduler*   scheduler();
    static cpu3300*     cpu();
    static term3315*    term(int n);
    static int          NumTerms();

    // turn speed regulation on (true) or off (false)
    static void RegulateCpuSpeed(bool realtime);

    // return true if the cpu speed is regulated
    static bool CpuSpeedRegulated();

    // simulate a few ms worth of instructions
    static void EmulateTimeslice();

    // because everything is static, we have to be told when
    // the sim is really starting and ending.
    static void Init();        // tee zero
    static void Cleanup();     // armageddon

    // Transfers blocks of data into memory.
    static void ReadByteProgram(const uint8 *pgm);

    // get/set current system configuration
    static SystemConfig* GetConfig();
    static void SetConfig(const SystemConfig &cfg);

private:
    // disallow copying
    System3300& operator=(const System3300& rhs);   // assignment
    System3300(System3300 &obj);                    // copy constructor

    // singleton members, and a flag to know when to know first time
    static bool               m_initialized;
    static SystemConfig      *m_config;         // active system config
    static Scheduler         *m_scheduler;      // for coordinating events in the emulator
    static cpu3300           *m_cpu;            // the central processing unit
    static vector<term3315*>  m_terms;          // a list of terminals in the system

    // these are used to control emulation speed throttling

    // stuff related to regulating simulation speed:
    static bool  m_regulated_speed; // 1=try to match real speed, 0=run full out

    static bool  m_firstslice;      // has m_realtimestart been initialized?
    static int64 m_realtimestart;   // wall time of when sim started
    static int   m_real_seconds;    // real time elapsed

    static unsigned long m_simsecs; // number of actual seconds simulated time elapsed

    // amount of actual simulated time elapsed, in ms
    static int64 m_simtime;

    // amount of adjusted simulated time elapsed, in ms because of user events
    // (eg, menu selection), sim time is often interrupted and we don't
    // actually desire to catch up.  m_simtime is the actual number of
    // simulated slices, while this var has been fudged to account for those
    // violations of realtime.
    static int64 m_adjsimtime;

    // keep a rolling average of how fast we are running in case it is reported.
    // we want to report the running average over the last second of realtime
    // to prevent the average from updating like crazy when running many times
    // realtime.
    static const int perf_hist_size;            // # of timeslices to track
    static     int64 perf_real_ms[100];         // realtime at start of each slice
    static       int perf_hist_len;             // number of entries written
    static       int perf_hist_ptr;             // next entry to write
};

#endif // _SYSTEM3300_H_
