
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "system3300.h"
#include "cpu3300.h"
#include "term3315.h"
#include "scheduler.h"
#include "bootstrap.h"
#include "dasm.h"
#include "UiSystem.h"

// ------------------------------------------------------------------------
// declare static members
// ------------------------------------------------------------------------

// declare the system singleton members
bool               System3300::m_initialized = false;
SystemConfig*      System3300::m_config      = NULL;
Scheduler*         System3300::m_scheduler   = NULL;
cpu3300*           System3300::m_cpu         = NULL;
vector<term3315*>  System3300::m_terms;

const int System3300::perf_hist_size = 100; // # of timeslices to track
int64     System3300::perf_real_ms[100];    // realtime at start of each slice
int       System3300::perf_hist_len;        // number of entries written
int       System3300::perf_hist_ptr;        // next entry to write

// stuff related to regulating simulation speed:
bool  System3300::m_regulated_speed;    // try to match speed of real system
bool  System3300::m_firstslice;         // has m_realtimestart been initialized?
int64 System3300::m_realtimestart;      // wall time of when sim started
int   System3300::m_real_seconds;       // real time elapsed

unsigned long System3300::m_simsecs; // number of actual seconds simulated time elapsed

// amount of actual simulated time elapsed, in ms
int64 System3300::m_simtime;

// amount of adjusted simulated time elapsed, in ms because of user events
// (eg, menu selection), sim time is often interrupted and we don't
// actually desire to catch up.  m_simtime is the actual number of
// simulated slices, while this var has been fudged to account for those
// violations of realtime.
int64 System3300::m_adjsimtime;

// ------------------------------------------------------------------------
// public members
// ------------------------------------------------------------------------

// constructor
System3300::System3300()
{
    // nothing to do
}


// destructor
System3300::~System3300()
{
    // do nothing; once the first instance is built, it is built for good
}


// initialize singleton
void
System3300::Init()
{
    ASSERT(!m_initialized);

    m_scheduler = new Scheduler();
    ASSERT(m_scheduler != NULL);

    SystemConfig ini_cfg = SystemConfig();   // get it from config file
    SetConfig(ini_cfg);

    // CPU speed regulation
    m_firstslice = true;
    m_simtime    = UiGetTimeMs();
    m_simsecs    = 0;
    m_adjsimtime = UiGetTimeMs();

    m_realtimestart = 0;    // wall time of when sim started
    m_real_seconds  = 0;    // real time elapsed

//  script_initpkg();           // in case we redirect to keyboard from a file

    m_initialized = true;
}


// return the current system configuration details
SystemConfig *
System3300::GetConfig()
{
    return m_config;
}


// build a system according to the spec.  if a system exists
// already, rebuild what parts are needed.
void
System3300::SetConfig(const SystemConfig &cfg)
{
    // that is, we don't need to rebuild if the only difference is
    // CPU speed regulation
    bool rebuild_required =
           (m_config == NULL)
        || (m_config->GetRamKB()             != cfg.GetRamKB())
        || (m_config->GetNumTerminals()      != cfg.GetNumTerminals())
        || (m_config->GetNumCassetteDrives() != cfg.GetNumCassetteDrives())
        ;

    if (!rebuild_required) {
        bool regulated = cfg.GetCpuSpeedRegulated();
        System3300::RegulateCpuSpeed(regulated);
        m_config->SetCpuSpeedRegulated(regulated);
        return;
    }

    // delete existing resources
    while (!m_terms.empty()) {
        term3315 *last = m_terms.back();
        m_terms.pop_back();
        delete last;
    }
    if (m_cpu != NULL)
        delete m_cpu;

    // save the new system configuration state
    if (m_config)
        delete m_config;
    m_config = new SystemConfig(cfg);

    // (re)build the CPU
    m_cpu = new cpu3300(*m_scheduler, cfg.GetRamKB());
    ASSERT(m_cpu != NULL);

    RegulateCpuSpeed(cfg.GetCpuSpeedRegulated());

    // add the requested number of terminals
    for(int i=0; i < cfg.GetNumTerminals(); i++)
        m_terms.push_back( new term3315(m_cpu, *m_scheduler, i) );
}


// because everything is static, the destructor does nothing, so we
// need this function to know when the real armageddon has arrived.
void
System3300::Cleanup(void)
{
//  script_closepkg();

    while(!m_terms.empty()) {
        delete m_terms.back();
        m_terms.pop_back();
    }

    delete m_cpu;
    m_cpu = NULL;

    delete m_scheduler;
    m_scheduler = NULL;

    m_config->Save();   // save config state for next start up
    delete m_config;
    m_config = NULL;
}


// singleton accessor
Scheduler*
System3300::scheduler()
{
    ASSERT(m_initialized);
    return m_scheduler;
}

cpu3300*
System3300::cpu() 
{
    ASSERT(m_initialized);
    return m_cpu;
}

term3315*
System3300::term(int n)
{
    ASSERT(m_initialized);
    return m_terms[n];
}


int
System3300::NumTerms()
{
    return m_terms.size();
}


// turn speed regulation on (true) or off (false)
void
System3300::RegulateCpuSpeed(bool realtime)
{
    m_regulated_speed = realtime;
    // reset the performance monitor history
    perf_hist_len = 0;
    perf_hist_ptr = 0;
}


// return true if the cpu speed is regulated
bool
System3300::CpuSpeedRegulated()
{
    return m_regulated_speed;
}


// simulate a few ms worth of instructions
void
System3300::EmulateTimeslice()
{
    // how many instructions to simulate before returning, in ms
    const int ts_ms = 16;  // 16 ms

    // for various reasons, we may be ahead or behind our target speed
    // in a given timeslice.  if we are off, we can simulate more/fewer
    // instructions in a a subsequent timeslice to compensate.  however,
    // we don't want to go overboard.  instead, we look at just the past
    // few timeslices to come up with a workload for the current timeslice.
    const int64 adj_window = 10*ts_ms;  // look at the last 10 timeslices

    uint64 now_ms = UiGetTimeMs();
    int64 realtime_elapsed;
    int64 offset;

    if (m_firstslice) {
        m_firstslice = false;
        m_realtimestart = now_ms;
    }
    realtime_elapsed = now_ms - m_realtimestart;
    offset = m_adjsimtime - realtime_elapsed;

    if (offset > adj_window) {
        // we're way ahead (probably because we are running unregulated)
        m_adjsimtime = realtime_elapsed + adj_window;
        offset = adj_window;
    } else if (offset < -adj_window) {
        // we've fallen way behind; catch up so we don't
        // run like mad after any substantial pause
        m_adjsimtime = realtime_elapsed - adj_window;
        offset = -adj_window;
    }

    if ((offset > 0) && CpuSpeedRegulated()) {

        // we are running ahead of schedule; kill some time.
        // we don't kill the full amount becuase the sleep function is
        // allowed to, and very well might, sleep longer than we asked.
        unsigned int ioffset = (unsigned int)(offset & (int64)0xFFF);       // bottom 4 sec or so
        UiSleep(ioffset/2);

    } else {

        // keep track of when each slice started
        perf_real_ms[perf_hist_ptr++] = now_ms;
        if (perf_hist_ptr >= perf_hist_size)
            perf_hist_ptr -= perf_hist_size;
        if (perf_hist_len < perf_hist_size)
            perf_hist_len++;

        // simulate one timeslice's worth of instructions
// FIXME: returns cycle count -- if this isn't needed, why return it?
        (void)cpu()->Run(ts_ms*625);    // 1.6 uS memory cycle time
        m_simtime    += ts_ms;
        m_adjsimtime += ts_ms;

        if (cpu()->GetCpuStatus() == cpu3300::CPUS_BADOP) {
            UiAlert("CPU attempted to execute an illegal op");
#if 0
            cpu()->Reset();  // force the reset
#endif
            return;
        }

        m_simsecs = (unsigned long)((m_simtime/1000) & 0xFFFFFFFF);

        int real_seconds_now = (int)(realtime_elapsed/1000);
        if (m_real_seconds != real_seconds_now) {
            m_real_seconds = real_seconds_now;
            if (perf_hist_len > 10) {
                // compute running performance average over the
                // last real second or so
                int n1 = (perf_hist_ptr - 1 + perf_hist_size)
                       % perf_hist_size;
                int64 ms_diff = 0;
                int slices = 0;
                for(int n=1; n<perf_hist_len; n+=10) {
                    int n0 = (n1 - n + perf_hist_size) % perf_hist_size;
                    slices = n;
                    ms_diff = (perf_real_ms[n1] - perf_real_ms[n0]);
                    if (ms_diff > 1000)
                        break;
                }
                float relative_speed = (float)(slices*ts_ms) / (float)ms_diff;

                // update the status bar with simulated seconds and performance
                UiSetSimSeconds(m_simsecs, relative_speed);
            }
        }

        // at least yield so we don't hog the whole machine
#if 0
        // when running unregulated speed mode and messing about with
        // the printer, sometimes there is an alert about wxYield
        // reentrancy.  Although this app says it doesn't care, the
        // printer code apparently doesn't set this flag thus the message.
        UiYield();
#else
        UiSleep(0);
#endif
    }
}


// ------------------------------------------------------------------------
// private members
// ------------------------------------------------------------------------

// Transfers blocks of data into memory.  It is used like this:
//     ReadByteProgram(bootstrap);
//     ReadByteProgram(tcboot);
// The program is a sequence of unsigned chars, encoded as blocks of
// contiguous data:
//     mm mm: length, in bytes; 0000 signals end of data
//     nn nn: load address
//     data[0], stored at address nnnn+0
//     data[1], stored at address nnnn+1
//     ...
//     data[mmmm-1], stored at address nnnn+mmmm-1
void
System3300::ReadByteProgram(const uint8 *pgm)
{
    System3300 sys;

    while(1) {

        // get next segment length
        uint16 length = pgm[0]*256 + pgm[1];
        pgm += 2;
        if (length == 0)
            break;      // no more segments

        // get load address
        uint16 load_addr = pgm[0]*256 + pgm[1];
        pgm += 2;

        // load bytes
        for(int addr=load_addr; addr<load_addr+length; addr++)
            sys.cpu()->SetMemory(addr, *pgm++);
    }
}


// =============================================================
// system configuration class
// =============================================================

// default constructor
SystemConfig::SystemConfig()
{
    Load(); // initialize the state based on the config file
}


// assignment
SystemConfig& SystemConfig::operator=(const SystemConfig &rhs)
{
    // just a straight copy suffices.
    // the default copy constructor would be OK, but I like it to be explicit.
    m_cpu_speed_regulated = rhs.m_cpu_speed_regulated;
    m_ram_kb              = rhs.m_ram_kb;
    m_num_terminals       = rhs.m_num_terminals;
    m_num_cassette_drives = rhs.m_num_cassette_drives;

    return *this;
}


// copy constructor
SystemConfig::SystemConfig(const SystemConfig &obj)
{
    *this = obj;
}


SystemConfig::~SystemConfig()
{
    // nothing to do
}


void
SystemConfig::SetCpuSpeedRegulated(bool v)
{
    m_cpu_speed_regulated = v;
}


void
SystemConfig::SetRamKB(int v)
{
    wxASSERT(v >= 4 && v <= 64);
    m_ram_kb = v;
}


void
SystemConfig::SetNumTerminals(int v)
{
    wxASSERT(v >= 1 && v <= 16);
    m_num_terminals = v;
}


void
SystemConfig::SetNumCassetteDrives(int v)
{
    wxASSERT(v >= 0 && v <= 16);
    wxASSERT((v & 1) == 0);
    m_num_cassette_drives = v;
}


// read from configuration file
void
SystemConfig::Load()
{
    wxString subgroup("..");

    bool bv;
    int v;

    ConfigReadBool(subgroup, "cpu_speed_regulated", &bv, true);
    SetCpuSpeedRegulated( bv );

    (void)ConfigReadInt(subgroup, "ram_kb", &v, 64);
    v = (v < 4) ? 4 : (v > 64) ? 64 : v;
    SetRamKB( v );

    (void)ConfigReadInt(subgroup, "terminals", &v, 1);
    v = (v < 1) ? 1 : (v > 16) ? 16 : v;
    SetNumTerminals( v );

    (void)ConfigReadInt(subgroup, "cassettes", &v, 2);
    v = (v + 1) & ~1;   // round up to next even (drives are in pairs)
    v = (v < 0) ? 0 : (v > 16) ? 16 : v;
    SetNumCassetteDrives( v );
}


// save to configuration file
void
SystemConfig::Save() const
{
    wxString subgroup("..");

    ConfigWriteBool(subgroup, "cpu_speed_regulated", GetCpuSpeedRegulated() );
    ConfigWriteInt( subgroup, "ram_kb",    GetRamKB() );
    ConfigWriteInt( subgroup, "terminals", GetNumTerminals() );
    ConfigWriteInt( subgroup, "cassettes", GetNumCassetteDrives() );
}

