// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2000, 2001
// Modified 2003 by Jim Battle for Wang Emu.
// Modified 2007 by Jim Battle for Emu3300.

// This file contains stuff localized to the core emulator.
// The interface part should not look at any of the internal
// organs which get exposed here.

#ifndef _SCHEDULER_H_
#define _SCHEDULER_H_

#include "w3300.h"  // pick up def of int32

// simple callback mechanism
#include "callback.h"

// for now, statically define how many timers can be active.
// in the future we could use a vector container to grow automatically.
#define NUM_TIMERS 20

// fwd reference
class Timer;

// this class manages event-driven behavior for the emulator.
// time advances every cpu tick, and callers can request to be called
// back with a specific parameter after some number of cycles.
// timers can also be killed before they have come due.
class Scheduler
{
public:
    Scheduler();
    ~Scheduler();

    // create a new timer; an int handle is returned.
    // this handle can be used to identify the timer later.
    // ticks is the number of clock ticks before the callback fires,
    // passing back the stored arg.
    template <class T, typename P>
    Timer* TimerCreate(int ticks, Callback<T,P> &fcn) {
        Callback<T,P> *fcncopy = new Callback<T,P>(fcn);
        return TimerCreateImpl(ticks, fcncopy);
    }

    // specify callback without and intermediate callback function, e.g.
    //   void TimerTestFoo::report(int i)
    //   { printf("got callback for timer %d after %d clocks\n", i, g_testtime); }
    //
    //   Timer* tmr = TimerCreate( 100, foo, &TimerTestFoo:report, 33 );
    //
    // After 100 clocks, TimerTestFoo::report(33) is called.
    template <class T, typename P>
    Timer* TimerCreate(int ticks, T& t, void (T::*f)(P), P p) {
        return TimerCreateImpl(ticks, new Callback<T,P>(t, f, p));
    }

    // remove a pending timer event by passing its index
    // (called only from Timer.Kill())
    void TimerKill(int n);

    // this is called periodically to advance time
    void TimerTick(int n);

private:
    // actual implementation
    Timer* TimerCreateImpl(int ticks, CallbackBase *fcn);

    // transfer accumulated timer deficit to each active timer.
    // n is added to the already elapsed time.
    // this shouldn't need to be called very frequently.
    void TimerCredit();

    // semaphore that indicates TimerCredit() is running so that
    // we can double check that nobody calls TimerTick() until
    // from a callback function.
    bool m_updating;

    // rather than updating all N counters every instruction, we instead
    // find the one which will expire first, and transfer that into
    // countdown and startcnt.  as simulated time ticks by, we decrement
    // countdown.
    //
    // when countdown goes <=0, all timers have (startcnt-countdown)
    // ticks removed from their own counter.
    int32 m_countdown;
    int32 m_startcnt;

    // any timer that has its own counter go <=0 will then
    // cause a callback to the supplied function using the
    // supplied parameters.
    struct {
        int32         ctr;      // counter until expiration
        CallbackBase *callback; // registered callback function
        Timer        *ptmr;     // timer handle
    } m_timer[NUM_TIMERS];

    // list of free timers
    int m_numFree;
    int m_freeIdx[NUM_TIMERS];

    // list of active timers
    int m_numActive;
    int m_activeIdx[NUM_TIMERS];

    // list of expired timers
    int m_numRetired;
    int m_retiredIdx[NUM_TIMERS];
};

// scale a floating point time in microseconds to be an
// argument appropriate for the TimerCreate() function.
// since each tick is one memory cycle, or 1.6 uS, we must scale appropriately.
#define TIMER_US(f) ((int)(0.625*(f)+0.5))
#define TIMER_MS(f) ((int)(625.0*(f)+0.5))

// this is used to kill off a timer that may or may not be inactive
#define ENSURE_TIMER_DEAD(thnd) {       \
        if ((thnd) != NULL) {           \
            (thnd)->Kill();             \
            (thnd) = NULL;              \
        }                               \
    } while (0)

// ======================================================================
// This is just a handle that Scheduler can pass back on timer creation,
// making it more natural for the recipient to manipulate the timer later.

class Timer
{
public:
    Timer(Scheduler *sched, int i) { s = sched; idx = i; };
    ~Timer() { };

    // kill off this timer
    void Kill();

// FIXME: expose it for now
    int idx;        // index into the array that Scheduler keeps
private:
    Scheduler *s;   // pointer to owning scheduler
};

#endif // ifdef _SCHEDULER_H_
