// This is a pretty simple set of routines to read, write, and check the
// validity of 8b intel hex files.  The format of these files are as follows:
//
//   field chars meaning
//   ----- ----- ----------------------------------------------------------
//      1     1  the start of record indicator, ":"
//      2     2  count of data bytes in this record, as two hex digits, ms first
//      3     4  address of first byte, as four hex digits, ms first
//      4     2  record type, as two hex digits, ms first
//                 00: data record
//                 01: end of file record.  the record must be exactly this:
//                        :00000001FF
//                 02: extended segment address record; disallowed
//                 03: start segment address record; disallowed
//                 04: extended linear address record; disallowed
//                 05: start linear address record; disallowed
//      5   0-2n 0 to n pairs of hex digits, each ms nibble first
//      6     2  two hex digit checksum, ms nibble first.  the value is chosen
//               such that the sum of bytes from all fields (except field 1),
//               mod 256, is zero.
//
// This implementation relaxes the standard to allow comment lines and
// comments trailing at the end of the line.  Comments are marked by
// a leading "-" character.

#ifndef _INCLUDE_INTEL_HEX_H_
#define _INCLUDE_INTEL_HEX_H_

#include <iostream>
#include <fstream>
#include <string>
using namespace std;

class HexFile
{
public:
    // mode can be 'r' for read, 'w' for write.
    // the max_record_len is meaningful only for 'w' mode;
    // it says how many data bytes are saved per record.
    HexFile(const char *filename, char filemode='r', int max_record_len=16);

    // flush any pending writes and close the file
    ~HexFile();

    // return true if named file was opened successfully
    bool Opened();

    // return true if the file is a legal .hex file;
    // this function is legal only if filemode is 'r'.
    // calling it also has the side effect of resetting the read pointer
    // to the start of file, so this function should be called immediately
    // after opening the file.
    bool IsLegal();

    // return true if the end of file has been reached.
    // legal only in 'r' mode.
    bool IsDone();

    // return the next data byte; the address ref is updated to correspond
    // to the address of the returned byte.
    // legal only in the 'r' mode.
    int GetNextByte(int &addr);

    // if there was a file error, return the line the error happened on
    int GetBadLine() { return m_curline; }

    // accept the next byte to be saved; legal only in the 'w' mode.
    // the byte is associated with the passed address.
    void SaveByte(int byte, int addr);

private:
    // in read mode, read in and decode next line
    bool parse_next_line();

    // in write mode, save any buffered bytes out to the file
    void flush();

    const string  m_filename;
    const char    m_filemode;         // 'r' for read, 'w' for write, 't' for test
    const int     m_max_record_len;   // in 'w' mode, maximum # data bytes per record
    fstream       m_file;             // file object

    int     m_curline;          // line number of current record
    bool    m_error;            // set if there is an error in the file format

    unsigned char m_data[256];  // 'r' mode: most recently read record
                                // 'w' mode: accumulate next record before flush()
    int     m_address;          // address associated with first entry of buffer
    int     m_data_size;        // number of valid bytes in m_data ('r' mode only)
    int     m_data_cnt;         // pointer to next m_data byte to read/write

    bool    m_prev_addr_valid;  // \__ saved so we can pack consecutive
    int     m_prev_addr;        // /   bytes into a single record.
};

#endif // _INCLUDE_INTEL_HEX_H_
