// This is a pretty simple set of routines to read, write, and check the
// validity of 8b intel hex files.

#include "intel_hex.h"
#include "w3300.h"

// constructor
HexFile::HexFile(const char *filename, char filemode, int max_record_len) :
    m_filename(filename),
    m_filemode(filemode),
    m_max_record_len(max_record_len),
    m_data_size(0),
    m_data_cnt(0),
    m_error(false)
{
    ASSERT(filemode == 'r' || filemode == 'w');
    ASSERT(filemode == 'r' || (max_record_len > 0 && max_record_len < 255));

    m_file.open(filename, (filemode == 'r') ? ios::in
                                            : ios::out | ios::trunc);
    m_curline = 0;
}


// destructor
HexFile::~HexFile()
{
    if (m_file.is_open()) {
        if (m_filemode == 'w') {
            flush();    // write out any pending data
            m_file << ":00000001ff" << endl;
        }
        m_file.close();
    }
}


// return true if the file was opened successfully
bool
HexFile::Opened()
{
    return m_file.is_open();
}


// return true if the file is a legal .hex file
bool
HexFile::IsLegal()
{
    ASSERT(m_filemode == 'r');

    // read each line and test error status until eof,
    // then rewind the file pointer
    bool bad_format = false;    // unless proven otherwise
    while (!IsDone()) {
        int a, b;
        b = GetNextByte(a);
        if (b < -1)
            bad_format = true;
        if (b < 0)
            break;      // error or EOF
    }

    if (bad_format)
        return false;

    // rewind to start of file
    m_file.seekp(ios_base::beg);
    m_curline = m_data_cnt = m_data_size = 0;

    return true;
}


// return true if the end of file has been reached.
bool
HexFile::IsDone()
{
    ASSERT(m_filemode == 'r');

    return m_file.eof();
}


// return the next data byte; the address ref is updated to correspond
// to the address of the returned byte.
// returns -1 if EOF; returns -2 if error, otherwise returns next byte.
// pretty ugly, I agree.
int
HexFile::GetNextByte(int &addr)
{
    ASSERT(m_filemode == 'r');

    if (m_data_size == m_data_cnt) {
        bool status = parse_next_line();
        if (!status)
            return -2;  // format error
        if (m_data_size == 0)
            return -1;  // EOF
    }

    addr = (m_address + m_data_cnt) & 0xFFFF;
    return static_cast<int>(m_data[m_data_cnt++]);
}


static int
hexvalue(int ch)
{
    unsigned char c = (unsigned char)ch;

    if (c >= '0' && c <= '9')
        return ch - '0';
    if (c >= 'A' && c <= 'F')
        return ch - 'A' + 10;
    if (c >= 'a' && c <= 'f')
        return c - 'a' + 10;

    ASSERT(0);  // caller didn't qualify input well enough
    return 0;   // keep compiler happy
}


// comments begin with a leading "-" character.
// squish out spaces and tabs along the way.
static void
remove_spaces_and_comments(char *buf)
{
    char *s, *d, ch;
    s = d = buf;
    while (ch = *s++) {
        if (ch == '-')
            break;
        if (ch != ' ' && ch != '\t')
            *d++ = ch;
    }
    *d = '\0';
}

// in read mode, read in and decode next line
// return false if the file isn't legal .hex format.
// note: EOF returns true.
bool
HexFile::parse_next_line()
{
    ASSERT(m_filemode == 'r');

    char buffer[600];
    int len;
    while (1) {
        if (m_file.eof())
            return true;    // EOF doesn't mean the file is bad

        m_file.getline(buffer, 600);
        m_curline++;

        remove_spaces_and_comments(buffer);
        len = strlen(buffer);

        if (len > 0)
            break;
    }

    m_error = true;     // guilty until proven innocent

    // must have at least a few fields
    if (len < 11)
        return false;

    // must have an odd number of characters per record
    if ((len & 1) != 1)
        return false;

    // check field 1
    if (buffer[0] != ':')
        return false;

    // check that the rest of the buffer contains legal hex digits,
    // converting pairs of ascii bytes into raw bytes in buffer[].
    int byte_count = 0;
    int sum = 0;
    for(int i=1; i<len; i += 2) {
        if (!isxdigit(buffer[i]) || !isxdigit(buffer[i+1]))
            return false;
        int b = 16*hexvalue(buffer[i]) + hexvalue(buffer[i+1]);
        buffer[byte_count++] = b;
        sum = (sum + b) & 0xFF;
    }
    if (sum != 0x00)
        return false;   // bad checksum

    // check field 2: count of the data bytes in this record
    if (buffer[0] != byte_count-5)
        return false;

    // field 3: address
    m_address = 256*(unsigned char)buffer[1] + (unsigned char)buffer[2];

    // field 4: check record type
    if (buffer[3] == 0x00) {
        // data record
        m_data_size = buffer[0];
        m_data_cnt = 0;         // no bytes read by client yet
        for(int i=0; i<m_data_size; i++)
            m_data[i] = buffer[i+4];    // +4 skips length, addr, field bytes
    } else if (buffer[3] == 0x01) {
        // end of file record; must be exactly :00000001FF
        if ((byte_count != 5) || (m_address != 0x0000))
            return false;
        m_data_size = 0;
    } else {
        return false;   // not a kind of hex file we can deal with
    }

    // everything went OK
    m_error = false;
    return true;
}


// accept the next byte to be saved.
// the byte is associated with the passed address.
void
HexFile::SaveByte(int byte, int addr)
{
    ASSERT(m_filemode == 'w');

    // make sure a record contains consecutive bytes
    if ((m_data_cnt > 0) && (addr != (m_address + m_data_cnt)))
        flush();

    // if start of line, establish the start address
    if (m_data_cnt == 0)
        m_address = addr;

    // add new byte
    m_data[m_data_cnt++] = byte;

    // pump it out if we've accumulated a full record
    if (m_data_cnt == m_max_record_len)
        flush();
}


// flush any pending write data
void
HexFile::flush()
{
    ASSERT(m_filemode == 'w');

    if (m_data_cnt > 0) {

        m_file.fill('0');

        // leader
        m_file.width(1);
        m_file << ':';

        // # data bytes in this record
        m_file.width(2);
        m_file << hex << m_data_cnt;

        // address of first byte of record
        m_file.width(4);
        m_file << m_address;

        // data record type byte
        m_file.width(2);
        m_file << 0x00; // record type

        // output data
        int sum = m_data_cnt + (m_address >> 8) + (m_address & 0xFF);
        for(int i=0; i<m_data_cnt; i++) {
            m_file.width(2);
            m_file << static_cast<int>(m_data[i]);
            sum += m_data[i];
        }

        // output checksum
        m_file.width(2);
        m_file << ((~sum + 1) & 0xff) << endl;

        m_curline++;
    }

    m_data_cnt = 0;
}
