// Disassembler for Wang 3300
// Jim Battle, 2005

#include <string.h>

#include "system3300.h"
#include "dasm.h"

// all characters in the string are copied as is, except for these which
// are expanded as described here:
//      i = expanded into 8b immediate as X'dd'
//      n = one's complemented and then treated like 'i' above
//      a = page 0/1 absolute reference, treated like 'i' for now
//      c = current page address relative, expanded as 16b X'dddd'
//      j = like c, but if the value is equal to *+2, don't print anything
//          used for optional jump files.  different from other types in that
//          it also affects the way the mnemonic is printed
//      s = (value of bits [1:0] + 1) are printed (1-4)
//          used for shift/rotate instructions
//      x = illegal opcode, and just X'dd' is printed
//      z = sign extend 8b immediate to 16b and print as X'dddd'
//  space = skip to column 8
//      , = skip to column 16

/*
page 98 (TABLE A.1) of the developers manual refers to these unknown ops:
    STG  (SKIP IF GENERAL TRUE)
    SFG  (SKIP IF GENERAL FALSE)
    ICH  (INITIATE CHANNEL)
    ECH  (END CHANNEL)

SFG  is also referenced in the same manual on page

STG and SFG appear on page 55 of wang3300.djvu from steve witham,
but they aren't explained.
*/


static char *encoded[256] = {

    /*0x00*/    "HLTj",
    /*0x01*/    "PARj",
    /*0x02*/    "Nj",   // appendix c lists this as "INj"
    /*0x03*/    "DNj",
    /*0x04*/    "AZAj",
    /*0x05*/    "XZAj",
    /*0x06*/    "TSAj",
    /*0x07*/    "TZAj",
    /*0x08*/    "SBj",
    /*0x09*/    "SBCj",
    /*0x0A*/    "SDj",
    /*0x0B*/    "LZAj",
    /*0x0C*/    "ONS i,; f",
    /*0x0D*/    "OFS n,; f",
    /*0x0E*/    "TASj",
    /*0x0F*/    "JZA",

    /*0x10*/    "STA i",
    /*0x11*/    "SFA n",
    /*0x12*/    "SAA i",
    /*0x13*/    "SMA n",
    /*0x14*/    "STS i,; f",
    /*0x15*/    "SFS n",
    /*0x16*/    "STG i",        // not sure if this is valid
    /*0x17*/    "SFG n",        // not sure if this is valid
    /*0x18*/    "AI i",
    /*0x19*/    "CI i",
    /*0x1A*/    "LAI i",
    /*0x1B*/    "LZI i",
    /*0x1C*/    "DLI z",
    /*0x1D*/    "BAI i",
    /*0x1E*/    "BXI i",
    /*0x1F*/    "BOI i",

    /*0x20*/    "SH s",
    /*0x21*/    "SHC s",
    /*0x22*/    "RT s",
    /*0x23*/    "RTC s",
    /*0x24*/    "JGT c",        // aka "JNC c"
    /*0x25*/    "JNE c",        // aka "JNZ c"
    /*0x26*/    "JLT c",
    /*0x27*/    "JEQ c",        // aka "JZ c"
    /*0x28*/    "x",
    /*0x29*/    "x",
    /*0x2A*/    "x",
    /*0x2B*/    "x",
    /*0x2C*/    "x",
    /*0x2D*/    "x",
    /*0x2E*/    "x",
    /*0x2F*/    "x",

    /*0x30*/    "STI i",
    /*0x31*/    "SFI n",
    /*0x32*/    "AKIj",
    /*0x33*/    "DSIj",
    /*0x34*/    "ICH?",         // initiate channel (not sure if this is right)
    /*0x35*/    "ECH?",         // end channel      (not sure if this is right)
    /*0x36*/    "x",
    /*0x37*/    "x",
    /*0x38*/    "TIAj",
    /*0x39*/    "ONMj",
    /*0x3A*/    "RDDj",
    /*0x3B*/    "WRDj",
    /*0x3C*/    "CIOj",
    /*0x3D*/    "x",
    /*0x3E*/    "x",
    /*0x3F*/    "x",

    /*0x40*/    "INC a", "INC* a", "INC c", "INC* c",
    /*0x44*/    "JMP a", "JMP* a", "JMP c", "JMP* c",
    /*0x48*/    "BA a",  "BA* a",  "BA c",  "BA* c",
    /*0x4C*/    "JST a", "JST* a", "JST c", "JST* c",

    /*0x50*/    "BX a",  "BX a",   "BX c",  "BX c",
    /*0x54*/    "JEI a", "JEI* a", "JEI c", "JEI* c",
    /*0x58*/    "BO a",  "BO* a",  "BO c",  "BO* c",
    /*0x5C*/    "x",     "x",      "x",     "x",

    /*0x60*/    "x",
    /*0x61*/    "x",
    /*0x62*/    "x",
    /*0x63*/    "x",
    /*0x64*/    "x",
    /*0x65*/    "x",
    /*0x66*/    "x",
    /*0x67*/    "x",
    /*0x68*/    "x",
    /*0x69*/    "x",
    /*0x6A*/    "x",
    /*0x6B*/    "x",
    /*0x6C*/    "x",
    /*0x6D*/    "x",
    /*0x6E*/    "x",
    /*0x6F*/    "x",

    /*0x70*/    "x",
    /*0x71*/    "x",
    /*0x72*/    "x",
    /*0x73*/    "x",
    /*0x74*/    "x",
    /*0x75*/    "x",
    /*0x76*/    "x",
    /*0x77*/    "x",
    /*0x78*/    "x",
    /*0x79*/    "x",
    /*0x7A*/    "x",
    /*0x7B*/    "x",
    /*0x7C*/    "x",
    /*0x7D*/    "x",
    /*0x7E*/    "x",
    /*0x7F*/    "x",

    /*0x80*/    "ADD a", "ADD* a", "ADD c", "ADD* c", "ADD+ a", "ADD- a", "ADD+ c", "ADD- c",
    /*0x88*/    "AC a",  "AC* a",  "AC c",  "AC* c",  "AC+ a",  "AC- a",  "AC+ c",  "AC- c",
    /*0x90*/    "C a",   "C* a",   "C c",   "C* c",   "C+ a",   "C- a",   "C+ c",   "C- c",
    /*0x98*/    "AMC a", "AMC* a", "AMC c", "AMC* c", "AMC+ a", "AMC- a", "AMC+ c", "AMC- c",
    /*0xA0*/    "DCM a", "DCM* a", "DCM c", "DCM* c", "DCM+ a", "DCM- a", "DCM+ c", "DCM- c",
    /*0xA8*/    "DAM a", "DAM* a", "DAM c", "DAM* c", "DAM+ a", "DAM- a", "DAM+ c", "DAM- c",
    /*0xB0*/    "DL a",  "DL* a",  "DL c",  "DL* c",  "DL+ a",  "DL- a",  "DL+ c",  "DL- c",
    /*0xB8*/    "DU a",  "DU* a",  "DU c",  "DU* c",  "DU+ a",  "DU- a",  "DU+ c",  "DU- c",
    /*0xC0*/    "LA a",  "LA* a",  "LA c",  "LA* c",  "LA+ a",  "LA- a",  "LA+ c",  "LA- c",
    /*0xC8*/    "UA a",  "UA* a",  "UA c",  "UA* c",  "UA+ a",  "UA- a",  "UA+ c",  "UA- c",
    /*0xD0*/    "LZ a",  "LZ* a",  "LZ c",  "LZ* c",  "LZ+ a",  "LZ- a",  "LZ+ c",  "LZ- c",
    /*0xD8*/    "UZ a",  "UZ* a",  "UZ c",  "UZ* c",  "UZ+ a",  "UZ- a",  "UZ+ c",  "UZ- c",
    /*0xE0*/    "XMA a", "XMA* a", "XMA c", "XMA* c", "XMA+ a", "XMA- a", "XMA+ c", "XMA- c",
    /*0xE8*/    "UAH a", "UAH* a", "UAH c", "UAH* c", "UAH+ a", "UAH- a", "UAH+ c", "UAH- c",
    /*0xF0*/    "x",     "x",      "x",     "x",      "x",      "x",      "x",      "x",
    /*0xF8*/    "x",     "x",      "x",     "x",      "x",      "x",      "x",      "x",
};


#define ARG_COLUMN     ( 6)     // which column arguments begin
#define COMMENT_COLUMN (16)     // which column comments begin

// given a 16b instruction, fill an external buffer with the disassembled
// version of that instruction.  returns 0 if OK, -1 if illegal operation.
int
dasm3300(uint16 addr, uint16 instr, char *buff, int p)
{
    const char *flags = "7PCZDV10";
    const bool p_known = (p == 0) || (p == 1);
    const uint8 op  = (instr >> 8) & 0xFF;
    uint8 imm = (instr >> 0) & 0xFF;
    const uint16 curaddr = (addr & 0xFF00) | imm;
    char *s = encoded[op];
    int len = 0;

    if (s[0] == 'x') {
        sprintf(buff, "DAC   X'%04X'", instr);
        return -1;
    }

    for( ; *s; s++) {

        switch (*s) {

            case 'a': // absolute page 0/1 address
		if (p_known) {
		    len += sprintf(buff+len, "X'0%c%02X'", (p) ? '1':'0', imm);
		} else {
		    len += sprintf(buff+len, "X'0?%02X'", imm);
		}
                break;

            case 'i': // 8b immediate
                len += sprintf(buff+len, "X'%02X'", imm);
                break;

            case 'n': // 8b immediate, complemented
                imm = (~imm) & 0xFF;
                len += sprintf(buff+len, "X'%02X'", imm);
                break;

            case 'c': // current page relative address
                len += sprintf(buff+len, "X'%04X'", curaddr);
                break;

            case 'j': // current page relative address, suppressed
                if (curaddr != addr+2) {
                    buff[len++] = 'J';
                    do { buff[len++] = ' '; } while (len < ARG_COLUMN);
                    len += sprintf(buff+len, "X'%04X'", curaddr);
                }
                break;

            case 's': // shift count
                len += sprintf(buff+len, "%d", (imm & 3)+1);
                break;

            case 'z': // 8b immediate sign extended to 16b
                {
                    uint16 imm16 = imm
                                 | ((imm & 0x80) ?  0xFF00 : 0x0000);
                    len += sprintf(buff+len, "X'%04X'", imm16);
                }
                break;

            case ' ': // tab to operand field
                do { buff[len++] = ' '; } while (len < ARG_COLUMN);
                break;

            case ',': // tab to comment field
                do { buff[len++] = ' '; } while (len < COMMENT_COLUMN);
                break;

            case 'f': // print immediate as decoded flags
		int k;
                for(k = 7; k >= 0; k--) {
                    if ((imm >> k) & 1)
                        buff[len++] = flags[7-k];
                }
                break;

            default:
                buff[len++] = *s;
                break;

        } // switch

        assert(len < DASM3300_MAX_LENGTH);

    } // for()

    // ensure termination
    buff[len] = '\0';

    return 0;
}
