#ifndef _CPU3300_H_
#define _CPU3300_H_

// Wang 3300 CPU
// Jim Battle, 2005

// this class models a Wang 3300 CPU.  It is populated with 64 KB of RAM.
// there are provisions for adding and removing I/O devices.

#include "w3300.h"
#include "system3300.h"
#include "IoDevice.h"

// forward class declarations
class Scheduler;

class cpu3300
{
public:

    // constructor
    cpu3300(Scheduler &scheduler, int max_ram_kb);

    // destructor
    ~cpu3300();

    // power on reset
    void Reset();

    // effect of hitting the front panel "Clear" button
    void Clear();

    // return the current state of the CPU
    enum cpus_t {
            CPUS_HALTED,        // front panel halt or HLT instruction hit
            CPUS_STEP,          // single step
            CPUS_EXQ,           // front panel EXQ mode
            CPUS_RUNNING,       // free running
            CPUS_BADOP,         // emulator state: hit illegal op
            CPUS_BREAKPOINT     // emulator state: hit breakpoint
            };
    cpus_t GetCpuStatus();
    void   SetCpuStatus(cpu3300::cpus_t state);

    // run for N memory cycles.
    // it might run a few extra since instructions take >1 cycle.
    // so by default with cycles=1, one instruction is executed.
    int Run(int cycles=1);

    // ----- I/O facilities -----

    // install an I/O device
    void AddIoDevice(IoDevice *dev, int station, int addr);

    // remove an I/O device
    void RemoveIoDevice(int addr);

    // method for an I/O device to change its interrupt request state
    void IoInterrupt(int addr, bool request=true);

    // ----- memory access -----
    void SetMemory(uint16 addr, uint8 data);
    uint8 GetMem8(uint16 addr, bool update_leds=false);
    uint16 GetMem16(uint16 addr);

    // "front panel" access
    enum { FPREG_B, FPREG_C, FPREG_Z, FPREG_A, FPREG_S, FPREG_M, FPREG_I };
    void SetReg(int reg, int value);
    int  GetReg(int reg);
    // used for time averaging LED intensity for GUI
    // GetLedHisto() returns the count of the number of times that combination
    // of 256 appeared since the last InitLedHisto().
    void InitLedHisto();
    int *GetLedHisto(int reg);  // return pointer to 256 entry array

    // ----- breakpointing facility -----
    int GetNumBreakpoints();
    int SetBreakpoint(uint16 addr, bool active);
    int GetBreakpoint(int n, uint16 *addr, bool *active);
    int ClearBreakpoint(uint16 addr, bool all=false);

    // return a string containing a dump of the register state.
    // it returns a pointer to a static buffer, so the returned string
    // should not be freed, and the routine is not re-entrant.
    char *RegState();

private:

    // ----- state twiddling routines -----

    // store the instruction execution time
    void SetCycles(int n) { m_cycle_cnt = n; };
    int  GetCycles()      { return m_cycle_cnt; };

    // perform an in-page jump
    void PageJump(uint8 off) { m_reg.BC = (m_reg.BC & 0xFF00) | off; };

    // perform a conditional skip
    void SkipIf(bool b) { m_reg.BC += (b) ? 4 : 2; };

    // change various flags
    void ChangeZ(uint8 v)    { m_reg.bZ = (v == 0x00); };
    void ChangeZ(uint16 v)   { m_reg.bZ = (v == 0x0000); };
    void ChangeV(uint8 v)    { m_reg.bV = ((v&0xF0)<0xA0) && ((v&0x0F)<0x0A); };
    void ChangeVIfD(uint8 v) { if (m_reg.bD) { ChangeV(v); } };
    void ChangeC(bool b)     { m_reg.bC = b; }
    void ChangeC(int v)      { m_reg.bC = (v != 0); }

    void EnableAllInterrupts()  { m_reg.bCP = false; };
    void DisableAllInterrupts() { m_reg.bCP = true; };

    // perform common effective address computation
    // passing even=true causes the address to be checked/forced even.
    // cycles is the instruction cycle count.
    int EffAddr4(uint8 op, uint8 imm, int cycles, bool even=false);

    // perform common effective address computation.
    // passing even=true causes the address to be checked/forced even.
    // cycles is the instruction cycle count.  besides being used for speed
    // regulation, this information is required to satisfy this restriction
    // noted on page 12 of the cpu manual:
    //    Memory reference instructions which reference a current page address
    //    and require more than two memory cycles for execution will not
    //    function properly when located in the last location of a page (XXFE).
    // inc indicates increment amount for pre/post inc/dec.
    int EffAddr8(uint8 op, uint8 imm, int cycles, bool even=false, int inc=1);

    // add together two 8b values with a carry state.
    // addition is done in decimal or binary based on m_reg.bD
    int Add8(int a, int b, int cy=0);

    // ----- I/O and interrupt state -----

    // these structures keep track of the (up to) eight stations that
    // area attached to the I/O bus and the (up to) 128 addressable
    // devices.  stations are numbered 1 to 8.
    //
    // the bus structure prioritizes things such that station[i] takes
    // priority over station[j] iff i<j.  Within a station, devices that
    // appear earlier in the interrupt chain take priority over those
    // that appear later.
    //
    // by analogy, I/O devices that are declared sooner for a given station
    // are given priority over those that appear later.  emulation still
    // honors that lower numbered stations are given priority.
    //
    // the structure below (and the algorithms that use it) is brute force to
    // keep things simple and since performance isn't an issue at this time.

    // get the status byte of the selected I/O device
    uint8 GetIoStatus(uint8 addr);

    // set the control byte of the selected I/O device
    void SetIoControl(uint8 addr, uint8 data);

    // get the read byte of the currently selected I/O device
    uint8 GetIoRead(uint8 addr);

    // set the write byte of the currently selected I/O device
    void SetIoWrite(uint8 addr, uint8 data);

    // set the per-station interrupt inhibit mask byte
    void SetIoIntMask(uint8 m);

    // indicate if there is a pending interrupt request
    bool IntPending() {
        return !m_reg.bCP && ((~m_reg.intMask & m_reg.intReq) != 0x00);
    };

    // return highest priority request device address
    // if there is no request, 0xFF is returned.
    uint8 GetIoRequestAddr();

    // the bus is 8b wide, but only 128 devices are allowed.
    // some devices qualify their decode by the MSB, which is
    // odd parity of the lower 7 bits.

    struct {

        // this keeps track of which attached devices are requesting service
        int numIntReq;          // number of devices requesting interrupt

        // this keeps track of the priority order of the I/O devices.
        // those at a lower index have higher priority.
        int numDevices;         // number of attached devices
        uint8 order[128];       // contains device address

    } m_station[8];

    // which I/O devices are requesting service
    bool m_intReqMap[128];

    // device object
    IoDevice *m_IoDevice[128];

    // this maps an I/O address to the station it is associated with
    int m_IoStation[128];

    // scheduling agent
    Scheduler &scheduler;

    // ----- register set -----
    struct {
        cpus_t state;   // run state

        uint8  A;       // primary accumulator
        uint8  Z;       // secondary accumulator; MSB of 16b accumulator
        uint16 BC;      // program counter
        uint8  M;       // memory transfer register
        uint8  I;       // instruction register (used only by front panel EXQ)

        bool   bS0;     // S reg, bit 0: user flag
        bool   bS1;     // S reg, bit 1: user flag
        bool   bV;      // S reg, bit 2: valid decimal result
        bool   bD;      // S reg, bit 3: decimal mode
        bool   bZ;      // S reg, bit 4: zero result
        bool   bC;      // S reg, bit 5: carry bit
        bool   bP;      // S reg, bit 6: absolute page
//      bool   bS7;     // S reg, bit 7: unused

        uint16 CHMN;    // DMA channel address

        uint8  intMask; // per-station interrupt inhibit (1=inhibit)
                        // bit 0 corresponds to station[0]
                        // and to "Station 1" in the manual.
        uint8  intReq;  // per-station interrupt request (active high)
        bool   bCP;     // global interrupt inhibit bit
    } m_reg;

    // ----- other state -----
    long  m_cycle_cnt;  // number of clock cycles consumed
    int   m_max_ram_kb; // size of RAM, in KB
    uint8 m_ram[65536]; // memory image

    // keep track of how often each register bit is high
    void TallyLedHisto(int clocks);
    int m_histo_cycles; // total number of cycles histo was collected over
    int m_histo_B[256];
    int m_histo_C[256];
    int m_histo_A[256];
    int m_histo_Z[256];
    int m_histo_S[256];
    int m_histo_M[256];
    int m_histo_bS0;
    int m_histo_bS1;
    int m_histo_bV;
    int m_histo_bD;
    int m_histo_bZ;
    int m_histo_bC;
    int m_histo_bP;
};

#endif // _CPU3300_H_
