// Wang 3300 CPU
// Jim Battle, 2005-2007

// this class models a Wang 3300 CPU.  It is populated with 64 KB of RAM.
// there are provisions for adding and removing I/O devices.

// the code sets m_reg.M for each byte read and calls the TallyLedHisto()
// function to make sure the panel state is updated.  Looking at the block
// diagram, it seems that M isn't updated with data being written (the write
// data comes from the "Z bus"), so writes don't cause an LED update.

#include "system3300.h"
#include "cpu3300.h"
#include "scheduler.h"
#include "dasm.h"
#include "UiSystem.h"  // for UiAlert()

// this value is used as a fill value for uninitialized locations
#define BAD_BYTE (0xCB)

// make sure the given value is even
#define ASSERT_EVEN(v) ASSERT(!((v) & 1))

// I had to disable this because it ws getting triggered; there was a chain
// of instructions where C was set, and a subsequent DAM was including the
// carry, leading to an odd pointer.  Specificially:
//
//    462C: JLT  X'4630'  (C is set & Z is not, so the branch is taken)
//    4630: JMP* X'4632'  (C flag is unaffected)
//    4410: DU   X'01C4'  (C flag is unaffected)
//    4412: DL   X'01C8'  (C flag is unaffected)
//    4414: DAM  X'01C4'  (C flag is added in; rslt=550B)
//    4416: DL*  X'01C4'  (pointer is odd)
#define WARN_ODD_POINTER 0

// constructor
cpu3300::cpu3300(Scheduler &scheduler, int max_ram_kb) :
    scheduler(scheduler),
    m_max_ram_kb(max_ram_kb)
{
    int n;

    // init per-device state
    for(n=0; n<128; n++) {
        m_intReqMap[n] = false;
        m_IoDevice[n]  = NULL;
        m_IoStation[n] = -1;    // not req'd, but better to init everything
    }

    // init per-station state
    for(int s=0; s<8; s++) {
        m_station[s].numIntReq  = 0;
        m_station[s].numDevices = 0;
        for(n=0; n<128; n++)
            m_station[s].order[n] = 0;  // not req'd, but init everything
    }

    // init other state
    m_cycle_cnt = 0;

    // clear ram
    for(int addr=0x0000; addr<0x10000; addr++)
        m_ram[addr] = BAD_BYTE;

    // reset CPU state
    Reset();

    // sample register state for later display
    InitLedHisto();
}


// destructor
cpu3300::~cpu3300()
{
    // no dynamic allocations, so nothing to do
}


// hardware reset
// in real life, probably most of these didn't get reset; it was up to
// the user to use the front panel to establish the proper values.
// however, in the interest of having known state to aid in debugging
// the emulator, everything gets set.
void
cpu3300::Reset()
{
    m_reg.state = CPUS_HALTED;  // run state

    m_reg.A     = 0x00;         // primary accumulator
    m_reg.Z     = 0x00;         // secondary accumulator; MSB of 16b accumulator
    m_reg.BC    = 0x0000;       // program counter
    m_reg.M     = 0x00;         // memory transfer register

    m_reg.bS0   = false;        // S m_reg, bit 0: user flag
    m_reg.bS1   = false;        // S m_reg, bit 1: user flag
    m_reg.bV    = false;        // S m_reg, bit 2: valid decimal result
    m_reg.bD    = false;        // S m_reg, bit 3: decimal mode
    m_reg.bZ    = false;        // S m_reg, bit 4: zero result
    m_reg.bC    = false;        // S m_reg, bit 5: carry bit
    m_reg.bP    = false;        // S m_reg, bit 6: absolute page
//  m_reg.bS7   = false;        // S m_reg, bit 7: unused

    m_reg.CHMN  = 0x0000;       // DMA channel address

    m_reg.intMask = 0xFF;       // inhibit all interrupts
    m_reg.intReq  = 0x00;       // no interrupt request
    DisableAllInterrupts();     // global interrupt inhibit

    // reset I/O devices
    Clear();
}


// front panel Clear button
// I'm not sure exactly what gets reset, but I do know that the register
// state (BC at least) doesn't get reset, as the procedure for running
// a program is to set BC, then press Clear, then press GO.
// Probably it mostly resets the I/O devices.
void
cpu3300::Clear()
{
    for(int n=0; n<128; n++) {
        if (m_IoDevice[n] != NULL)
            m_IoDevice[n]->Reset();
    }
}


// install an I/O device
void
cpu3300::AddIoDevice(IoDevice *dev, int station, int addr)
{
    ASSERT(station >= 0 && station < 8);
    ASSERT(addr >= 0 && addr < 128);
    ASSERT(m_IoDevice[addr] == NULL);

    m_IoDevice[addr]  = dev;
    m_IoStation[addr] = station;

    // add device to tail end of priority list
    m_station[station].order[m_station[station].numDevices++] = (uint8)addr;
}


// remove an I/O device
void
cpu3300::RemoveIoDevice(int addr)
{
    ASSERT(addr >= 0 && addr < 128);
    ASSERT(m_IoDevice[addr] != NULL);

    m_IoDevice[addr]  = NULL;
    int station = m_IoStation[addr];
    m_IoStation[addr] = -1;     // not required, but do it anyway

    // remove the device from the priority list at that station
    // it is not necessarily at the end.
    int devs = m_station[station].numDevices;
    ASSERT(devs > 0);

    for (int n = 0; n < devs; n++) {
        if (m_station[station].order[n] == addr) {
            // found it -- now delete that entry, shifting the rest back
            for(int m=n; m < n-1; m++)
                m_station[station].order[m] = m_station[station].order[m+1];
            m_station[station].numDevices--;
            return;
        }
    }
    // should have found it in the list somewhere
    ASSERT(0);
}


FILE *g_outf = NULL;

// method for an I/O device to change its interrupt request state
void
cpu3300::IoInterrupt(int addr, bool request)
{
    ASSERT(addr >= 0 && addr < 128);
if (g_outf)
fprintf(g_outf, "IoInterrupt(addr=%02X, req=%d\n", addr, request);

    // only take note if the state from this device has changed
    if (request != m_intReqMap[addr]) {

        // update at device level
        m_intReqMap[addr] = request;

        // update at station level
        int station = m_IoStation[addr];
        if (request) {
            m_station[station].numIntReq++;
            // indicate station's desire to interrupt,
            // although it may be inhibited for other reasons
            m_reg.intReq |= (1 << station);
if (g_outf)
fprintf(g_outf, "station %d adding to interrupt chain: %02X\n", station, m_reg.intReq);
        } else {
            ASSERT(m_station[station].numIntReq > 0);
            m_station[station].numIntReq--;
            if (m_station[station].numIntReq == 0) {
                // last request from this station dropped
                m_reg.intReq &= ~(1 << station);
if (g_outf)
fprintf(g_outf, "station %d removing from interrupt chain: %02X\n", station, m_reg.intReq);
            }
        }
    }
}


// get the status byte of the currently selected I/O device
uint8
cpu3300::GetIoStatus(uint8 addr)
{
    uint8 devaddr = (addr & 0x7F);
    IoDevice *dev = m_IoDevice[devaddr];

    ASSERT(dev != NULL);
    if (dev == NULL)
        return BAD_BYTE;

    int data;
    data = dev->RdStatus(devaddr);

    return (uint8)data;
}


// set the control byte of the currently selected I/O device
void
cpu3300::SetIoControl(uint8 addr, uint8 data)
{
    uint8 devaddr = (addr & 0x7F);
    IoDevice *dev = m_IoDevice[devaddr];

#if 0
    ASSERT(dev != NULL);
#else
    if (dev == NULL)
        UiAlert("BC=%04X: SetIoControl to unmapped device (0x%02X)",
		m_reg.BC, devaddr);
#endif
    if (dev == NULL)
        return;

    dev->WrControl(devaddr, data);
}


// get the read byte of the currently selected I/O device
uint8
cpu3300::GetIoRead(uint8 addr)
{
    uint8 devaddr = (addr & 0x7F);
    IoDevice *dev = m_IoDevice[devaddr];

    ASSERT(dev != NULL);
    if (dev == NULL)
        return BAD_BYTE;

    int data;
    data = dev->RdData(devaddr);

    return (uint8)data;
}


// set the write byte of the currently selected I/O device
void
cpu3300::SetIoWrite(uint8 addr, uint8 data)
{
    uint8 devaddr = (addr & 0x7F);
    IoDevice *dev = m_IoDevice[devaddr];

    ASSERT(dev != NULL);
    if (dev == NULL)
        return;

    dev->WrData(devaddr, data);
}


// set the per-station interrupt inhibit mask byte.
// bit 0 corresponds to m_station[0], which corresponds to what the manual
// describes a "Station 1".  A one bit inhibits the station from
// interrupting, and a zero bit allows interrupts from that station.
void
cpu3300::SetIoIntMask(uint8 m)
{
    // it has changed
    m_reg.intMask = m;
}


// return highest priority request device address.
// if there is no request, 0xFF is returned.
uint8
cpu3300::GetIoRequestAddr()
{
    // we can't use IntPending() since this is qualified by bCP,
    // which was automatically cleared upon recognition of the interrupt.
    if (((~m_reg.intMask & m_reg.intReq) == 0x00)) {
        // FIXME: what does the real hardware return?
        UiAlert("BC=%04X: GetIoRequestAddr() called with no pending interrupt",
		m_reg.BC);
        return 0xFF;    // no request pending
    }

    // pick lowest requesting station
    uint8 req = (m_reg.intReq & ~m_reg.intMask);
    int station = (req & 0x01) ? 0
                : (req & 0x02) ? 1
                : (req & 0x04) ? 2
                : (req & 0x08) ? 3
                : (req & 0x10) ? 4
                : (req & 0x20) ? 5
                : (req & 0x40) ? 6
                               : 7;

    // now scan devices of that station by priority
    int addr = -1;
    for(int n=0; n<m_station[station].numDevices; n++) {
        int devaddr = m_station[station].order[n];
        if (m_intReqMap[devaddr]) {
            addr = devaddr;
            break;
        }
    }
    ASSERT(addr >= 0);

    return (uint8)addr;
}


// add together two 8b values with a carry state.
// addition is done in decimal or binary based on m_reg.bD
int
cpu3300::Add8(int a, int b, int cy)
{
    if (m_reg.bD) {
        // decimal
        int a0 = ((a>>0) & 0xF);
        int a1 = ((a>>4) & 0xF);
        int b0 = ((b>>0) & 0xF);
        int b1 = ((b>>4) & 0xF);
#if 1
        // make sure digits are BCD
        ASSERT(a0<10 && a1<10 && b0<10 && b1<10);
#endif
        int s0 = a0 + b0 + cy;
        int s1 = a1 + b1;
        int s2 = 0;
        if (s0 > 9) { s0 -= 10; s1++; }
        if (s1 > 9) { s1 -= 10; s2++; }
        return (s2<<8) + (s1<<4) + s0;
    } else {
        // binary
        return a + b + cy;
    }
}


// perform common effective address computation.
//
// passing even=true causes the address to be checked/forced even.
//
// cycles is the instruction cycle count.  besides being used for speed
// regulation, this information is required to satisfy this restriction
// noted on page 12 of the cpu manual:
//    Memory reference instructions which reference a current page address
//    and require more than two memory cycles for execution will not
//    function properly when located in the last location of a page (XXFE).
// This is probably because BC increments by 2 at the end of the second
// cycle, such that if the effective address is calculated after this,
// BC will be pointing to the next page.
//
// even indicates whether the resulting effective address must be even.
int
cpu3300::EffAddr4(uint8 op, uint8 imm, int cycles, bool even)
{
    uint16 abs_addr = ((m_reg.bP) ? 0x0100 : 0x0000) | imm;
    uint16 cur_addr = (m_reg.BC & 0xFF00) | imm;
    uint16 rv, rv0;

    switch (op & 3) {
        case 0: // absolute page, direct
            rv = abs_addr;
            SetCycles(cycles);
            break;
        case 1: // absolute page, indirect
            rv0 = abs_addr;
            ASSERT_EVEN(rv0);
            rv0 &= ~1;
            rv = GetMem16(rv0);
            SetCycles(cycles+2);
            break;
        case 2: // current page, direct
            rv = cur_addr;
            ASSERT((cycles <= 2) || ((m_reg.BC & 0x00FE) != 0xFE));
            SetCycles(cycles);
            break;
        case 3: // current page, indirect
            rv0 = cur_addr;
            ASSERT_EVEN(rv0);
            rv0 &= ~1;
            ASSERT((cycles <= 2) || ((m_reg.BC & 0x00FE) != 0xFE));
            rv = GetMem16(rv0);
            SetCycles(cycles+2);
            break;
    }

    if (even) {
#if 1
        if (WARN_ODD_POINTER && ((rv & 1) != 0))
            UiAlert("EffAddr4 not even: 0x%04X, at BC=0x%04X", rv, m_reg.BC);
#else
        ASSERT_EVEN(rv);
#endif
        rv &= ~1;  // enforce evenness
    }

    return rv;
}


// perform common effective address computation.
//
// passing even=true causes the address to be checked/forced even.
//
// cycles is the instruction cycle count.  besides being used for speed
// regulation, this information is required to satisfy this restriction
// noted on page 12, described above before EffAddr4.
//
// inc indicates increment amount for pre/post inc/dec.
int
cpu3300::EffAddr8(uint8 op, uint8 imm, int cycles, bool even, int inc)
{
    uint16 rv0, rv, rvinc;

    switch (op & 7) {
        case 0: // absolute page, direct
            rv = ((m_reg.bP) ? 0x0100 : 0x0000) | imm;
            SetCycles(cycles);
            break;
        case 1: // absolute page, indirect
            rv0 = ((m_reg.bP) ? 0x0100 : 0x0000) | imm;
            ASSERT_EVEN(rv0);
            rv0 &= ~1;
            rv = GetMem16(rv0);
            SetCycles(cycles+2);
            break;
        case 2: // current page, direct
            rv = (m_reg.BC & 0xFF00) | imm;
            ASSERT((cycles <= 2) || ((m_reg.BC & 0x00FE) != 0xFE));
            SetCycles(cycles);
            break;
        case 3: // current page, indirect
            rv0 = (m_reg.BC & 0xFF00) | imm;
            ASSERT_EVEN(rv0);
            rv0 &= ~1;
            ASSERT((cycles <= 2) || ((m_reg.BC & 0x00FE) != 0xFE));
            rv = GetMem16(rv0);
            SetCycles(cycles+2);
            break;
        case 4: // absolute page, indirect, auto increment
            rv0 = ((m_reg.bP) ? 0x0100 : 0x0000) | imm;
            ASSERT_EVEN(rv0);
            rv0 &= ~1;
            rv = GetMem16(rv0);
            rvinc = rv + inc;
            SetMemory(rv0+0, (uint8)(rvinc >> 8));
            SetMemory(rv0+1, (uint8)(rvinc >> 0));
            SetCycles(cycles+2);
            break;
        case 5: // absolute page, indirect, auto decrement
            rv0 = ((m_reg.bP) ? 0x0100 : 0x0000) | imm;
            ASSERT_EVEN(rv0);
            rv0 &= ~1;
            rv = GetMem16(rv0);
            rv -= inc;
            SetMemory(rv0+0, (uint8)(rv >> 8));
            SetMemory(rv0+1, (uint8)(rv >> 0));
            SetCycles(cycles+2);
            break;
        case 6: // current page, indirect, auto increment
            rv0 = (m_reg.BC & 0xFF00) | imm;
            ASSERT_EVEN(rv0);
            rv0 &= ~1;
            ASSERT((cycles <= 2) || ((m_reg.BC & 0x00FE) != 0xFE));
            rv = GetMem16(rv0);
            rvinc = rv + inc;
            SetMemory(rv0+0, (uint8)(rvinc >> 8));
            SetMemory(rv0+1, (uint8)(rvinc >> 0));
            SetCycles(cycles+2);
            break;
        case 7: // current page, indirect, auto decrement
            rv0 = (m_reg.BC & 0xFF00) | imm;
            ASSERT_EVEN(rv0);
            rv0 &= ~1;
            ASSERT((cycles <= 2) || ((m_reg.BC & 0x00FE) != 0xFE));
            rv = GetMem16(rv0);
            rv -= inc;
            SetMemory(rv0+0, (uint8)(rv >> 8));
            SetMemory(rv0+1, (uint8)(rv >> 0));
            SetCycles(cycles+2);
            break;
    }

    if (even) {
#if 1
        if (WARN_ODD_POINTER && ((rv & 1) != 0))
            UiAlert("EffAddr8 not even: 0x%04X, at BC=0x%04X", rv, m_reg.BC);
#else
        ASSERT_EVEN(rv);
#endif
        rv &= ~1;  // enforce evenness
    }

    return rv;
}


// set CPU register state
void
cpu3300::SetReg(int reg, int value)
{
    switch (reg) {
        case FPREG_B:
            m_reg.BC = (m_reg.BC & 0x00FF) | (value << 8);
            break;
        case FPREG_C:
            m_reg.BC = (m_reg.BC & 0xFF00) | (value << 0);
            break;
        case FPREG_A:
            m_reg.A = value;
            break;
        case FPREG_Z:
            m_reg.Z = value;
            break;
        case FPREG_S:
            m_reg.bS0 = (value & 0x01) ? true : false;
            m_reg.bS1 = (value & 0x02) ? true : false;
            m_reg.bV  = (value & 0x04) ? true : false;
            m_reg.bD  = (value & 0x08) ? true : false;
            m_reg.bZ  = (value & 0x10) ? true : false;
            m_reg.bC  = (value & 0x20) ? true : false;
            m_reg.bP  = (value & 0x40) ? true : false;
            break;
        case FPREG_M:
            m_reg.M = value;
            break;
        case FPREG_I:
            m_reg.I = value;
            break;
        default:
            assert(0);
    }
}


// get CPU register state
int
cpu3300::GetReg(int reg)
{
    switch (reg) {
        case FPREG_B:
            return (m_reg.BC >> 8) & 0xFF;
        case FPREG_C:
            return (m_reg.BC >> 0) & 0xFF;
        case FPREG_A:
            return m_reg.A;
        case FPREG_Z:
            return m_reg.Z;
        case FPREG_S:
            return (m_reg.bS0 ? 0x01 : 0x00) |
                   (m_reg.bS1 ? 0x02 : 0x00) |
                   (m_reg.bV  ? 0x04 : 0x00) |
                   (m_reg.bD  ? 0x08 : 0x00) |
                   (m_reg.bZ  ? 0x10 : 0x00) |
                   (m_reg.bC  ? 0x20 : 0x00) |
                   (m_reg.bP  ? 0x40 : 0x00) ;
        case FPREG_M:
            return m_reg.M;
        case FPREG_I:
            return m_reg.I;
        default:
            assert(0);
            return 0;
    }
}


// clear out register state counters, used for GUI LED display
void
cpu3300::InitLedHisto()
{
    m_histo_cycles = 0;

    for(int n=0; n<256; n++) {
        m_histo_B[n] =
        m_histo_C[n] =
        m_histo_A[n] =
        m_histo_Z[n] =
        m_histo_S[n] =
        m_histo_M[n] = 0;
    }

    m_histo_bS0 = 0;
    m_histo_bS1 = 0;
    m_histo_bV  = 0;
    m_histo_bD  = 0;
    m_histo_bZ  = 0;
    m_histo_bC  = 0;
    m_histo_bP  = 0;

    // initialize to current cpu state, in case the CPU is stopped,
    // so that the GUI LEDs reflect the current state
    TallyLedHisto(1);
}


// accumulate how long each register bit is in the '1' state,
// and also the total number of clocks we've tallied.
void
cpu3300::TallyLedHisto(int clocks)
{
    m_histo_cycles += clocks;
    m_histo_B[(m_reg.BC >> 8) & 0xff] += clocks;
    m_histo_C[(m_reg.BC >> 0) & 0xff] += clocks;
    m_histo_A[m_reg.A] += clocks;
    m_histo_Z[m_reg.Z] += clocks;
    m_histo_M[m_reg.M] += clocks;
    m_histo_bS0 += (m_reg.bS0) ? clocks : 0;
    m_histo_bS1 += (m_reg.bS1) ? clocks : 0;
    m_histo_bV  += (m_reg.bV)  ? clocks : 0;
    m_histo_bD  += (m_reg.bD)  ? clocks : 0;
    m_histo_bZ  += (m_reg.bZ)  ? clocks : 0;
    m_histo_bC  += (m_reg.bC)  ? clocks : 0;
    m_histo_bP  += (m_reg.bP)  ? clocks : 0;
}


// return an int array with array[0] representing the percent of the time
// that bit 0 of the requested register was high, array[1] the percent of
// the time bit 1 was high, etc.
int *
cpu3300::GetLedHisto(int reg)
{
    // percent high time for each of the 8b of the specified register
    static int percent[8];
    int *hist;

    switch (reg) {
        case FPREG_B:
            hist = m_histo_B;
            break;
        case FPREG_C:
            hist = m_histo_C;
            break;
        case FPREG_A:
            hist = m_histo_A;
            break;
        case FPREG_Z:
            hist = m_histo_Z;
            break;
        case FPREG_S:
            m_histo_S[0x01] = m_histo_bS0;
            m_histo_S[0x02] = m_histo_bS1;
            m_histo_S[0x04] = m_histo_bV;
            m_histo_S[0x08] = m_histo_bD;
            m_histo_S[0x10] = m_histo_bZ;
            m_histo_S[0x20] = m_histo_bC;
            m_histo_S[0x40] = m_histo_bP;
            hist = m_histo_S;
            break;
        case FPREG_M:
            hist = m_histo_M;
            break;
        default:
            ASSERT(0);
            return NULL;   // return something to keep compiler happy
    }

    // not figure out the percent of the time that a given bit is high
    for(int b=0; b<8; b++)
        percent[b] = 0;

    for(int i=0; i<256; i++) {
        int h = hist[i];
        for(int b=0; b<8; b++) {
            if (i & (1 << b))
                percent[b] += h;
        }
    }

    ASSERT(m_histo_cycles != 0);
    for(int b=0; b<8; b++)
        percent[b] = (int)(100.0*percent[b] / m_histo_cycles);

    return percent;
}


// return a string containing a dump of the register state.
// it returns a pointer to a static buffer, so the returned string
// should not be freed, and the routine is not re-entrant.
char *
cpu3300::RegState()
{
    const char *flags = "7PCZDV10";
    static char buff[100];
    int len = sprintf(buff, "BC=%04X, Z=%02X, A=%02X, flags=",
                        m_reg.BC, m_reg.Z, m_reg.A);

//  buff[len++] = (m_reg.bS7) ? '7' : '.';
    buff[len++] = (m_reg.bP)  ? 'P' : '.';
    buff[len++] = (m_reg.bC)  ? 'C' : '.';
    buff[len++] = (m_reg.bZ)  ? 'Z' : '.';
    buff[len++] = (m_reg.bD)  ? 'D' : '.';
    buff[len++] = (m_reg.bV)  ? 'V' : '.';
    buff[len++] = (m_reg.bS1) ? '1' : '.';
    buff[len++] = (m_reg.bS0) ? '0' : '.';
    buff[len]   = '\0';

    return buff;
}


static int
dasm_one(cpu3300 *pcpu, uint16 addr, int p)
{
    static bool init = false;
    if (!init) {
        init = true;
        g_outf = fopen("dasm.txt", "w");
    }

    char *regs = pcpu->RegState();

    char buffer[DASM3300_MAX_LENGTH];
    uint8 b0 = pcpu->GetMem8(addr+0);
    uint8 b1 = pcpu->GetMem8(addr+1);
    uint16 instr = (b0 << 8) + b1;
    int drc = dasm3300( addr, instr, buffer, p );
    fprintf(g_outf, "%s     %04X: %04X  %s\n", regs, addr, instr, buffer);
    fflush(g_outf);

    return drc;
}


// run N memory cycles and return.
// the routine may run for a few extra cycles because instructions take
// two to five cycles to complete, so we can't promise to hit the count
// exactly.  so, for example, by requesting cycles=1, one instruction
// will be performed.  we return the actual number of cycles run.
// ISSUE: is bit 7 of the S register implemented like bits 0 and 1?
// ISSUE: when executing a non-jump instruction at page off 0xFE, does it
//        go to the next page or wrap to the start of the current page?
// ISSUE: how to ICH and ECH work?
// ISSUE: what do STG and SFG test?

bool g_dasm = false;

int
cpu3300::Run(int cycles)
{
    int cycle_count = 0;

    while (cycle_count < cycles) {

        // check for breakpoints
        // FIXME

        if (IntPending()) {

            // an interrupt is pending
            DisableAllInterrupts();
            SetMemory(0x0000, (uint8)(m_reg.BC >> 8));
            SetMemory(0x0001, (uint8)(m_reg.BC >> 0));
            m_reg.BC = 0x0002;
            TallyLedHisto(2);
            SetCycles(2);

        } else if ((m_reg.state == CPUS_HALTED) ||
                   (m_reg.state == CPUS_BADOP) ||
                   (m_reg.state == CPUS_BREAKPOINT)) {

            // halted state (HALT, or BADOP)
            // just let some simulated time pass
            // we want to let I/O devices to get processing done, in
            // case the CPU is halted waiting for an interrupt from one.
            TallyLedHisto(2);
            SetCycles(2);

        } else {
             // state is one of CPUS_RUNNING, CPUS_STEP, CPUS_EXQ

            ASSERT_EVEN(m_reg.BC);    // program counter must be even
            m_reg.BC &= ~1;           // enforce evenness

#if 0
    // from TCLOAD
            if (m_reg.BC == 0x05BA) {
                m_reg.BC = 0x6FCA;  // jump to HLT; trap here to load BASIC
            }
#endif

            if (m_reg.BC == 0x4200)
                g_dasm = false;  // start disassembling at start of BASIC interp

            if (g_dasm)
                (void)dasm_one(this, m_reg.BC, m_reg.bP ? 1:0);

            uint8 op, imm;      // the two bytes of the instruction
            if (m_reg.state == CPUS_EXQ) {
                op  = m_reg.I;
                imm = m_reg.M;
            } else {
                // fetch instruction normally
                op  = GetMem8(m_reg.BC,     true);
                imm = GetMem8(m_reg.BC + 1, true);
            }

            switch (op) {

            case 0x00: // HLTJ -- halt and jump after halt
                // ISSUE: what breaks the spell of HLT?
                //        I assume an interrupt can do it,
                //        and of course front panel activity too.
                // ISSUE: when this instruction is hit, should BC remain pointing
                //        at the HLTJ, is it incremented, or is it already set
                //        to point at the target address?  I'm assuming the last.
                m_reg.state = CPUS_HALTED;
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x01: // PARJ -- compute parity
                // ISSUE: the docs say that bit 7 must be 0 to begin with.
                //        what if it isn't?  I'm going with the assumption
                //        that it inverts the parity.
                // ISSUE: the docs say "The odd parity bit is generated in bit 7
                //        of the A register based on bits 0-6 of the A register."
                //        Is this odd parity including bit 7?  I'm assuming not.
                { int t1 = m_reg.A ^ (m_reg.A << 4);
                  int t2 = t1 ^ (t1 << 2);
                  int t3 = t2 ^ (t2 << 1);
                  m_reg.A = (m_reg.A & 0x7F) | (~t3 & 0x80);
                }
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x02: // NJ -- complement A and jump
                if (m_reg.bD) {
                    m_reg.A = ((0x90 - (m_reg.A & 0xF0)) & 0xF0)
                            | ((0x09 - (m_reg.A & 0x0F)) & 0x0F);
                    ChangeV(m_reg.A);
                } else {
                    m_reg.A = ~m_reg.A;
                }
                ChangeZ(m_reg.A);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x03: // DNJ -- complement {Z,A} and jump
                if (m_reg.bD) {
                    m_reg.A = ((0x90 - (m_reg.A & 0xF0)) & 0xF0)
                            | ((0x09 - (m_reg.A & 0x0F)) & 0x0F);
                    m_reg.Z = ((0x90 - (m_reg.Z & 0xF0)) & 0xF0)
                            | ((0x09 - (m_reg.Z & 0x0F)) & 0x0F);
                } else {
                    m_reg.A = ~m_reg.A;
                    m_reg.Z = ~m_reg.Z;
                }
                ChangeZ(m_reg.A);       // odd, but yes, the spec says A register
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x04: // AZAJ -- add Z to A and Jump
                {
                    int sum = Add8( m_reg.A, m_reg.Z, (m_reg.bC)?1:0 );
                    m_reg.A  = (uint8)sum;
                    ChangeZ(m_reg.A);
                    ChangeVIfD(m_reg.A);
                    ChangeC(sum >= 256);
                }
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x05: // XZAJ -- exhange Z and A and jump
                {
                    uint8 tmp = m_reg.A;
                    m_reg.A = m_reg.Z;
                    m_reg.Z = tmp;
                }
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x06: // TSAJ -- transfer S to A and jump
                m_reg.A = ((m_reg.bS0) ? 0x01 : 0x00)
                        | ((m_reg.bS1) ? 0x02 : 0x00)
                        | ((m_reg.bV)  ? 0x04 : 0x00)
                        | ((m_reg.bD)  ? 0x08 : 0x00)
                        | ((m_reg.bZ)  ? 0x10 : 0x00)
                        | ((m_reg.bC)  ? 0x20 : 0x00)
                        | ((m_reg.bP)  ? 0x40 : 0x00)
                //      | ((m_reg.bS7) ? 0x80 : 0x00)
                        ;
                ChangeZ(m_reg.A);
                ChangeVIfD(m_reg.A);
                PageJump(imm);
                SetCycles(3);           // yes, CPU manual says three clocks
                TallyLedHisto(1);       // account for extra execution cycle
                break;

            case 0x07: // TZAJ -- transfer Z to A and jump
                m_reg.A = m_reg.Z;
                ChangeZ(m_reg.A);
                ChangeVIfD(m_reg.A);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x08: // SBJ -- shift binary double and jump
                m_reg.Z = ((m_reg.Z << 1) & 0xFE) | (m_reg.A >= 128);
                m_reg.A = ((m_reg.A << 1) & 0xFE);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x09: // SBCJ -- shift binary double with carry and jump
                m_reg.Z = ((m_reg.Z << 1) & 0xFE) | (m_reg.A >= 128);
                m_reg.A = ((m_reg.A << 1) & 0xFE) | (m_reg.bC ? 1:0);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x0A: // SDJ -- shift decimal double and jump
                m_reg.Z = ((m_reg.Z << 4) & 0xF0) | ((m_reg.A >> 4) & 0x0F);
                m_reg.A = ((m_reg.A << 4) & 0xF0);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x0B: // LZAJ -- load via {Z,A} and jump
                {
                    uint16 addr = (m_reg.Z << 8) | m_reg.A;
                    ASSERT_EVEN(addr);
                    addr &= ~1;
                    m_reg.Z = GetMem8(addr+0, true);
                    m_reg.A = GetMem8(addr+1, true);
                }
                PageJump(imm);
                SetCycles(4);
                break;

            case 0x0C: // ONS
                m_reg.bS0 |= ((imm & 0x01) != 0x00);
                m_reg.bS1 |= ((imm & 0x02) != 0x00);
                m_reg.bV  |= ((imm & 0x04) != 0x00);
                m_reg.bD  |= ((imm & 0x08) != 0x00);
                m_reg.bZ  |= ((imm & 0x10) != 0x00);
                m_reg.bC  |= ((imm & 0x20) != 0x00);
                m_reg.bP  |= ((imm & 0x40) != 0x00);
    //      m_reg.bS7 |= ((imm & 0x80) != 0x00);
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x0D: // OFS
                m_reg.bS0 &= ((imm & 0x01) != 0x00);
                m_reg.bS1 &= ((imm & 0x02) != 0x00);
                m_reg.bV  &= ((imm & 0x04) != 0x00);
                m_reg.bD  &= ((imm & 0x08) != 0x00);
                m_reg.bZ  &= ((imm & 0x10) != 0x00);
                m_reg.bC  &= ((imm & 0x20) != 0x00);
                m_reg.bP  &= ((imm & 0x40) != 0x00);
    //      m_reg.bS7 &= ((imm & 0x80) != 0x00);
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x0E: // TASJ -- transfer A to S and jump
                m_reg.bS0 = ((m_reg.A & 0x01) != 0x00);
                m_reg.bS1 = ((m_reg.A & 0x02) != 0x00);
                m_reg.bV  = ((m_reg.A & 0x04) != 0x00);
                m_reg.bD  = ((m_reg.A & 0x08) != 0x00);
                m_reg.bZ  = ((m_reg.A & 0x10) != 0x00);
                m_reg.bC  = ((m_reg.A & 0x20) != 0x00);
                m_reg.bP  = ((m_reg.A & 0x40) != 0x00);
    //      m_reg.bS7 = ((m_reg.A & 0x80) != 0x00);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x0F: // JZA -- jump via {Z,A}
                m_reg.BC = (m_reg.Z << 8) | m_reg.A;
                ASSERT_EVEN(m_reg.BC);
                m_reg.BC &= ~1;  // enforce evenness
                SetCycles(2);
                break;

            case 0x10: // STA -- skip if true A
                SkipIf((m_reg.A & imm) == imm);
                SetCycles(2);
                break;

            case 0x11: // SFA -- skip if false A
                SkipIf((m_reg.A & ~imm) == 0x00);
                SetCycles(2);
                break;

            case 0x12: // SAA -- skip if any A
                SkipIf((m_reg.A & imm) != 0x00);
                SetCycles(2);
                break;

            case 0x13: // SMA -- skip if mixed A
                SkipIf((m_reg.A | imm) != 0xFF);
                SetCycles(2);
                break;

            case 0x14: // STS -- skip if true S
                {
                    bool skip = (m_reg.bS0 | ((imm & 0x01) == 0))
                              & (m_reg.bS1 | ((imm & 0x02) == 0))
                              & (m_reg.bV  | ((imm & 0x04) == 0))
                              & (m_reg.bD  | ((imm & 0x08) == 0))
                              & (m_reg.bZ  | ((imm & 0x10) == 0))
                              & (m_reg.bC  | ((imm & 0x20) == 0))
                              & (m_reg.bP  | ((imm & 0x40) == 0))
    //                    & (m_reg.bS7 | ((imm & 0x80) == 0))
                              ;
                    SkipIf(skip);
                }
                SetCycles(2);
                break;

            case 0x15: // SFS -- skip if false S
                {
                    bool skip = (!m_reg.bS0 | ((imm & 0x01) != 0))
                              & (!m_reg.bS1 | ((imm & 0x02) != 0))
                              & (!m_reg.bV  | ((imm & 0x04) != 0))
                              & (!m_reg.bD  | ((imm & 0x08) != 0))
                              & (!m_reg.bZ  | ((imm & 0x10) != 0))
                              & (!m_reg.bC  | ((imm & 0x20) != 0))
                              & (!m_reg.bP  | ((imm & 0x40) != 0))
    //                    & (!m_reg.bS7 | ((imm & 0x80) != 0))
                              ;
                    SkipIf(skip);
                }
                SetCycles(2);
                break;

            case 0x16: // STG?
            case 0x17: // SFG?
                m_reg.state = CPUS_BADOP;       // FIXME
                break;

            case 0x18: // AI -- add immediate to A
                m_reg.A = (uint8)Add8( m_reg.A, imm );
                ChangeZ(m_reg.A);
                ChangeVIfD(m_reg.A);
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x19: // CI -- compare immediate to A
                // ISSUE: the instruction description says the Z and C are
                //        affected, but the associated flags summary box
                //        shows C,V,Z affected.  I'm assuming V is affected
                //        if in decimal mode.
                {
                    int sum = (m_reg.A ^ 0xFF) + imm + 1;
                    ChangeZ((uint8)sum);
                    ChangeVIfD((uint8)sum);
                    ChangeC(sum >= 256);
                }
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x1A: // LAI -- load A immediate
                m_reg.A = imm;
                ChangeZ(m_reg.A);
                ChangeVIfD(m_reg.A);
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x1B: // LZI -- load Z immediate
                m_reg.Z = imm;
                ChangeZ(m_reg.Z);
                ChangeVIfD(m_reg.Z);
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x1C: // DLI -- double load immediate
                // ISSUE: summary box says the V is affected, but the written
                //        description doesn't mention it.  I'm assuming V
                //        isn't affected since any negative number would not
                //        produce a valid extension.
                m_reg.A = imm;
                m_reg.Z = (imm & 0x80) ? 0xFF : 0x00;
                ChangeZ(m_reg.A);
                ChangeC(0);
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x1D: // BAI -- boolean AND immediate
                m_reg.A = m_reg.A & imm;
                ChangeZ(m_reg.A);
                ChangeVIfD(m_reg.A);
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x1E: // BXI -- boolean XOR immediate
                m_reg.A = m_reg.A ^ imm;
                ChangeZ(m_reg.A);
                ChangeVIfD(m_reg.A);
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x1F: // BOI -- boolean OR immediate
                m_reg.A = m_reg.A | imm;
                ChangeZ(m_reg.A);
                ChangeVIfD(m_reg.A);
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x20: // SH -- shift A left
                {
                    int sh = (imm & 3) + 1;
                    m_reg.A = (uint8)(m_reg.A << sh);
                }
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x21: // SHC -- shift A left with carry
                {
                    int sh = (imm & 3) + 1;
                    m_reg.A = (uint8)(  (m_reg.A         << (sh  )) |
                                       ((m_reg.bC ? 1:0) << (sh-1)) );
                    ChangeC(0);
                }
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x22: // RT -- rotate A left
                {
                    int sh = (imm & 3) + 1;
                    m_reg.A = (uint8)(((m_reg.A << (  sh)) & 0xFF) |
                                      ((m_reg.A >> (8-sh)) & 0xFF));
                }
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x23: // RTC -- rotate A left with carry
                {
                    int sh  = (imm & 3) + 1;
                    int tmp = m_reg.A | (m_reg.bC ? 0x100 : 0x000);
                    tmp = ((tmp << (  sh)) & 0x1FF)
                        | ((tmp >> (9-sh)) & 0x1FF);
                    m_reg.A = (uint8)tmp;
                    ChangeC(tmp > 0xFF);
                }
                m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x24: // JGT -- jump if greater than
                if (!m_reg.bC)
                    PageJump(imm);
                else
                    m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x25: // JNE -- jump if not equal
                if (!m_reg.bZ)
                    PageJump(imm);
                else
                    m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x26: // JLT -- jump if less than
                if (m_reg.bC && !m_reg.bZ)
                    PageJump(imm);
                else
                    m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x27: // JEQ -- jump if equal
                if (m_reg.bZ)
                    PageJump(imm);
                else
                    m_reg.BC += 2;
                SetCycles(2);
                break;

            case 0x30: // STI -- skip if true I/O
                SkipIf((GetIoStatus(m_reg.Z) & imm) == imm);
                SetCycles(2);
                break;

            case 0x31: // SFI -- skip if false I/O
                SkipIf((GetIoStatus(m_reg.Z) & ~imm) == 0x00);
                SetCycles(2);
                break;

            case 0x32: // AKIJ -- acknowledge interrupt and jump
                // ISSUE: what does the real hardware do if AKIJ is called
                //        when there is no pending request?
                m_reg.A = GetIoRequestAddr();
                {
                    int devaddr = m_reg.A & 0x7F;
                    IoDevice *dev = m_IoDevice[devaddr];
                    // ASSERT(dev != NULL);
                    if (dev == NULL)
                        UiAlert("BC=%04X: AKIJ to unmapped device 0x%02X",
				m_reg.BC, m_reg.A);
                    if (dev != NULL)
                        dev->IAck(devaddr);
                }
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x33: // DSIJ -- disable interrupts and jump
                DisableAllInterrupts();
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x34: // ICH? -- initiate channel
            case 0x35: // ECH? -- end channel
                m_reg.state = CPUS_BADOP;       // FIXME
                break;

            case 0x38: // TIAJ -- transfer I/O to A and jump
                m_reg.A = GetIoStatus(m_reg.Z);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x39: // ONMJ -- on interrupt mask and jump
                SetIoIntMask(m_reg.A);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x3A: // RDDJ -- read data and jump
                m_reg.A = GetIoRead(m_reg.Z);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x3B: // WRDJ -- write data and jump
                SetIoWrite(m_reg.Z, m_reg.A);
                PageJump(imm);
                SetCycles(2);
                break;

            case 0x3C: // CIOJ -- control signal to I/O and jump
                SetIoControl(m_reg.Z, m_reg.A);
                PageJump(imm);
                SetCycles(2);
                break;

            // INC -- increment memory location
            case 0x40: case 0x41: case 0x42: case 0x43:
                {
                    uint16 eaddr = EffAddr4(op, imm, 3, false);
                    uint8 val = GetMem8(eaddr, true) + 1;
                    SetMemory(eaddr, val);
                    ChangeZ(val);
                    ChangeVIfD(val);
                }
                m_reg.BC += 2;
                break;

            // JEI -- jump and enable interrupts
            case 0x54: case 0x55: case 0x56: case 0x57:
                EnableAllInterrupts();
                // ... fall through to JMP below

            // JMP -- jump to effective address
            case 0x44: case 0x45: case 0x46: case 0x47:
                m_reg.BC = EffAddr4(op, imm, 2, true);
                ASSERT_EVEN(m_reg.BC);
                m_reg.BC &= ~1;  // enforce evenness
                break;

            // BA -- boolean AND memory to A
            case 0x48: case 0x49: case 0x4A: case 0x4B:
                {
                    uint16 eaddr = EffAddr4(op, imm, 3);
                    int mem = GetMem8(eaddr, true);
                    m_reg.A &= mem;
                    ChangeZ(m_reg.A);
                    ChangeVIfD(m_reg.A);
                }
                m_reg.BC += 2;
                break;

            // JST -- jump and store location
            case 0x4C: case 0x4D: case 0x4E: case 0x4F:
                // ISSUE: what happens if the effective address is (xx)FE?
                //        should we jump to (xx)00 or (xx+1)00?
                // read disclaimer on page -12-; it is specific but odd
                {
                    uint16 eaddr = EffAddr4(op, imm, 2, true);
                    ASSERT_EVEN(eaddr);
                    eaddr &= ~1;  // enforce evenness
                    SetMemory(eaddr+0, (uint8)((m_reg.BC+2) >> 8));
                    SetMemory(eaddr+1, (uint8)((m_reg.BC+2) >> 0));
                    m_reg.BC = eaddr + 2;
                }
                break;

            // BX -- boolean XOR memory to A
            case 0x50: case 0x51: case 0x52: case 0x53:
                {
                    uint16 eaddr = EffAddr4(op, imm, 3);
                    int mem = GetMem8(eaddr, true);
                    m_reg.A ^= mem;
                    ChangeZ(m_reg.A);
                    ChangeVIfD(m_reg.A);
                }
                m_reg.BC += 2;
                break;

            // BO -- boolean OR memory to A
            case 0x58: case 0x59: case 0x5A: case 0x5B:
                {
                    uint16 eaddr = EffAddr4(op, imm, 3);
                    int mem = GetMem8(eaddr, true);
                    m_reg.A |= mem;
                    ChangeZ(m_reg.A);
                    ChangeVIfD(m_reg.A);
                }
                m_reg.BC += 2;
                break;

            // ADD -- add memory to A
            case 0x80: case 0x81: case 0x82: case 0x83:
            case 0x84: case 0x85: case 0x86: case 0x87:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    int mem  = GetMem8(eaddr, true);
                    int rslt = Add8(m_reg.A, mem);
                    m_reg.A = (uint8)rslt;
                    ChangeZ(m_reg.A);
                    ChangeVIfD(m_reg.A);
                }
                m_reg.BC += 2;
                break;

            // AC -- add memory with carry to A
            case 0x88: case 0x89: case 0x8A: case 0x8B:
            case 0x8C: case 0x8D: case 0x8E: case 0x8F:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    int mem  = GetMem8(eaddr, true);
                    int rslt = Add8(m_reg.A, mem, (m_reg.bC) ? 1:0);
                    m_reg.A = (uint8)rslt;
                    ChangeC(rslt >= 256);
                    ChangeZ(m_reg.A);
                    ChangeVIfD(m_reg.A);
                }
                m_reg.BC += 2;
                break;

            // C -- compare A to memory
            // ISSUE: the instruction description says the Z and C are
            //        affected, but the associated flags summary box
            //        shows C,V,Z affected.  I'm assuming V is affected
            //        if in decimal mode.
            case 0x90: case 0x91: case 0x92: case 0x93:
            case 0x94: case 0x95: case 0x96: case 0x97:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    int mem  = GetMem8(eaddr, true);
                    int rslt = (m_reg.A ^ 0xFF) + mem + 1;
                    ChangeC(rslt >= 256);
                    ChangeZ((uint8)rslt);
                    ChangeVIfD((uint8)rslt);
                }
                m_reg.BC += 2;
                break;

            // AMC -- add A with carry to memory
            case 0x98: case 0x99: case 0x9A: case 0x9B:
            case 0x9C: case 0x9D: case 0x9E: case 0x9F:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    int mem  = GetMem8(eaddr, true);
                    int rslt = Add8(m_reg.A, mem, (m_reg.bC) ? 1:0);
                    SetMemory(eaddr, (uint8)rslt);
                    ChangeC(rslt >= 256);
                    ChangeZ((uint8)rslt);
                    ChangeVIfD((uint8)rslt);
                }
                m_reg.BC += 2;
                break;

            // DCM -- double byte compare to memory
            case 0xA0: case 0xA1: case 0xA2: case 0xA3:
            case 0xA4: case 0xA5: case 0xA6: case 0xA7:
                {
                    uint16 eaddr = EffAddr8(op, imm, 4, true, 2);
                    ASSERT_EVEN(eaddr);
                    eaddr &= ~1;  // enforce evenness
                    int mem  = GetMem16(eaddr);
                    int rslt = ((~((m_reg.Z << 8) | m_reg.A)) & 0xFFFF)
                               + mem + 1;
                    ChangeZ((uint16)rslt);
                    ChangeC(rslt > 0xFFFF);
                    ChangeVIfD( (uint8)(rslt >> 8) );  // only high byte counts
                }
                m_reg.BC += 2;
                break;

            // DAM -- double byte add to memory
            case 0xA8: case 0xA9: case 0xAA: case 0xAB:
            case 0xAC: case 0xAD: case 0xAE: case 0xAF:
                {
                    uint16 eaddr = EffAddr8(op, imm, 4, true, 2);
                    ASSERT_EVEN(eaddr);
                    eaddr &= ~1;  // enforce evenness
                    int memMS = GetMem8(eaddr+0, true);
                    int memLS = GetMem8(eaddr+1, true);
                    int rslt0 = Add8(m_reg.A, memLS, (m_reg.bC)   ? 1:0);
                    int rslt1 = Add8(m_reg.Z, memMS, (rslt0>=256) ? 1:0);
                    SetMemory(eaddr+1, (uint8)rslt0);
                    SetMemory(eaddr+0, (uint8)rslt1);
                    ChangeZ((uint8)(rslt0 | rslt1));
                    ChangeC(rslt1 >= 256);
                    ChangeVIfD((uint8)rslt1);
                }
                m_reg.BC += 2;
                break;

            // DL -- double load
            case 0xB0: case 0xB1: case 0xB2: case 0xB3:
            case 0xB4: case 0xB5: case 0xB6: case 0xB7:
                {
                    uint16 eaddr = EffAddr8(op, imm, 4, true, 2);
                    ASSERT_EVEN(eaddr);
                    eaddr &= ~1;  // enforce evenness
                    m_reg.Z = GetMem8(eaddr+0, true);
                    m_reg.A = GetMem8(eaddr+1, true);
                    ChangeZ((uint8)(m_reg.Z | m_reg.A));
                    ChangeVIfD((uint8)m_reg.Z);
                }
                m_reg.BC += 2;
                break;

            // DU -- double unload
            case 0xB8: case 0xB9: case 0xBA: case 0xBB:
            case 0xBC: case 0xBD: case 0xBE: case 0xBF:
                {
                    uint16 eaddr = EffAddr8(op, imm, 4, true, 2);
                    ASSERT_EVEN(eaddr);
                    eaddr &= ~1;  // enforce evenness
                    SetMemory((eaddr+0), m_reg.Z);
                    SetMemory((eaddr+1), m_reg.A);
                }
                m_reg.BC += 2;
                break;

            // LA -- load A
            case 0xC0: case 0xC1: case 0xC2: case 0xC3:
            case 0xC4: case 0xC5: case 0xC6: case 0xC7:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    m_reg.A = GetMem8(eaddr, true);
                    ChangeZ(m_reg.A);
                    ChangeVIfD(m_reg.A);
                }
                m_reg.BC += 2;
                break;

            // UA -- unload A
            case 0xC8: case 0xC9: case 0xCA: case 0xCB:
            case 0xCC: case 0xCD: case 0xCE: case 0xCF:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    SetMemory(eaddr, m_reg.A);
                }
                m_reg.BC += 2;
                break;

            // LZ -- load Z
            case 0xD0: case 0xD1: case 0xD2: case 0xD3:
            case 0xD4: case 0xD5: case 0xD6: case 0xD7:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    m_reg.Z = GetMem8(eaddr, true);
                    ChangeZ(m_reg.Z);
                    ChangeVIfD(m_reg.Z);
                }
                m_reg.BC += 2;
                break;

            // UZ -- unload Z
            case 0xD8: case 0xD9: case 0xDA: case 0xDB:
            case 0xDC: case 0xDD: case 0xDE: case 0xDF:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    SetMemory(eaddr, m_reg.Z);
                }
                m_reg.BC += 2;
                break;

            // XMA -- exchange memory and A
            case 0xE0: case 0xE1: case 0xE2: case 0xE3:
            case 0xE4: case 0xE5: case 0xE6: case 0xE7:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    uint8 mem = GetMem8(eaddr, true);
                    SetMemory(eaddr, m_reg.A);
                    m_reg.A = mem;
                    ChangeZ(m_reg.A);
                    ChangeVIfD(m_reg.A);
                }
                m_reg.BC += 2;
                break;

            // UAH -- unload A high digit
            case 0xE8: case 0xE9: case 0xEA: case 0xEB:
            case 0xEC: case 0xED: case 0xEE: case 0xEF:
                {
                    uint16 eaddr = EffAddr8(op, imm, 3);
                    uint8 mem = GetMem8(eaddr, true);
                    uint8 merge = (m_reg.A & 0xF0) | (mem & 0x0F);
                    SetMemory(eaddr, merge);
                }
                m_reg.BC += 2;
                break;

            // undefined instructions
            case 0x28: case 0x29: case 0x2A: case 0x2B:
            case 0x2C: case 0x2D: case 0x2E: case 0x2F:
                                  case 0x36: case 0x37:
                       case 0x3D: case 0x3E: case 0x3F:
            case 0x5C: case 0x5D: case 0x5E: case 0x5F:
            case 0x60: case 0x61: case 0x62: case 0x63:
            case 0x64: case 0x65: case 0x66: case 0x67:
            case 0x68: case 0x69: case 0x6A: case 0x6B:
            case 0x6C: case 0x6D: case 0x6E: case 0x6F:
            case 0x70: case 0x71: case 0x72: case 0x73:
            case 0x74: case 0x75: case 0x76: case 0x77:
            case 0x78: case 0x79: case 0x7A: case 0x7B:
            case 0x7C: case 0x7D: case 0x7E: case 0x7F:
            case 0xF0: case 0xF1: case 0xF2: case 0xF3:
            case 0xF4: case 0xF5: case 0xF6: case 0xF7:
            case 0xF8: case 0xF9: case 0xFA: case 0xFB:
            case 0xFC: case 0xFD: case 0xFE: case 0xFF:
                m_reg.state = CPUS_BADOP;
                break;

            default:
                ASSERT(0);  // all should have been covered
                m_reg.state = CPUS_BADOP;
                break;
            }

            // both EXQ and STEP turn into halt after one instruction
            if (m_reg.state != CPUS_RUNNING)
                m_reg.state = CPUS_HALTED;

        } // end of CPUS_RUNNING case

        // advance the scheduler by the duration of the instruction
        cycle_count += GetCycles();
        scheduler.TimerTick(GetCycles());
    }

    return cycle_count;
}


// return the current state of the CPU
cpu3300::cpus_t
cpu3300::GetCpuStatus()
{
    return m_reg.state;
}

// set the current state of the CPU
void
cpu3300::SetCpuStatus(cpu3300::cpus_t state)
{
    m_reg.state = state;
}


// fetch 8b word from memory, optionally updating the M register LEDs as we go
uint8
cpu3300::GetMem8(uint16 addr, bool update_leds)
{
    if ((addr >> 10) >= m_max_ram_kb) {
        ASSERT(0);
        return 0x00;
    }

    if (update_leds)
        TallyLedHisto(1);

    m_reg.M = m_ram[addr];

    if (g_dasm) {
        fprintf(g_outf, "   M[%04X] == %02X\n", addr, m_ram[addr]);
        fflush(g_outf);
    }

    return m_reg.M;
};


// fetch 16b word from memory, updating the M register LEDs as we go
uint16
cpu3300::GetMem16(uint16 addr)
{
    ASSERT_EVEN(addr);
    uint8 msb = GetMem8(addr + 0, true);
    uint8 lsb = GetMem8(addr + 1, true);
    return (uint16)((msb<<8) + lsb);
}


// set a byte of memory to the specified value
void
cpu3300::SetMemory(uint16 addr, uint8 data)
{
    if ((addr >> 10) >= m_max_ram_kb) {
        ASSERT(0);
        return;
    }

    if (g_dasm) {
        fprintf(g_outf, "   M[%04X] <- %02X (%02X)\n", addr, data, m_ram[addr]);
        fflush(g_outf);
    }

    m_ram[addr] = data;
}
