#ifndef _INCLUDE_WANGTAPE_H_
#define _INCLUDE_WANGTAPE_H_

// This class is used to represent a virtual tape file for the Wang 3300
// emulator.  It could be useful for a utility external to its use in the
// emulator.
//
// The format in the file is as follows.  The uint16 fields are stored
// little endian format:
//
//      struct tape_header_t {
//          char   magic[28] = "Wang Virtual Cassette<CR><LF>";
//          uint8  version   = 0x01; // file format version
//          uint8  readonly;         // 0x01=read only; 0x00=writable
//          uint16 model;            // either 2200 or 3300
//          uint16 label_len = N;    // length of the following label
//          char   label[N];         // label text
//      };
//      The label is N bytes long, straight ascii, 0x00 terminated.
//      Lines are separated by either <CR>, <LF>, or a <CR><LF> pair.
//
//      The rest of the tape is a bit stream.  Even bits correspond to
//      transitions in track 0, and odd bits correspond to transitions in
//      track 1.  The tape is read and written a bit pair (corresponding
//      even and odd bits) at a time.
//
// The logical tape length is fixed at the time the virtual tape image file
// is created.  While it would be possible to add a mechanism to change the
// tape length, it is purposefully avoided to keep things simple.
//
// The virtual tape is single sided, just like a real Wang data tape.
// It also helps reduce confusion.
//
// The tape image is maintained in memory.  All changes to its state
// are only to the in-memory image.  The Flush() method must be called in
// order to make these changes permanent.  The destructor does NOT
// automatically flush dirty state, so make sure Flush() is called first
// if that is important.
//
// The tape has ~1 sec of leader and trailer.
// This is not stored in the file, but is determined by the position.
// If position is < N bit times, then it is BOT.
// If position is > tape_length+N bit times, then it is EOT.

#include "w3300.h"

#include <vector>
using std::vector;
#include <string>
using std::string;

class WangVirtualTape
{
public:
    // create new file of specified length
    explicit WangVirtualTape(int length_kb, int model=2200);
    // open existing file
    explicit WangVirtualTape(string filepath);

    ~WangVirtualTape();         // destructor

    // save any updated state; if filepath is "", then save to
    // current filename, otherwise rename and save it to that file.
    void Flush(string filepath="");

    // erase the data on the tape -- the label is unchanged
    void Erase();

    // return path to file on disk
    bool IsNamed() const;       // has name been assigned?
    string Filename() const;    // if yes, the path to the file

    // although it doesn't affect this library, log whether this is
    // for the Wang 2200 or 3300.
    int Model() const;
    void SetModel(int model);

    // return or set the label associated with the tape.
    // the label can't be longer than 64K characters.
    string Label() const;
    void SetLabel(string label);

    // a tape can be write protected, or not
    bool IsReadOnly() const;
    void SetReadOnly(bool);

    // indicates the state of the in-memory tape image may have
    // changed since it was created.
    bool IsDirty() const;

    // tape size, in bytes or approx minutes
    int Length() const;           // in bits

    // set read/write offset relative to start of tape, in bits
    int  Offset() const;
    void SetOffset(int);

    // status of tape at position under r/w head
    bool IsGap() const;         // apparently between data blocks
    bool IsBot() const;         // at beginning of tape
    bool IsEot() const;         // at end of tape

    // data manipulation
    enum wvt_t { wvt_gap=0, wvt_zero, wvt_one, wvt_both };
    wvt_t ReadBit();            // get current bit and advance
    void  WriteBit(wvt_t);      // write bit and advance

private:
    // file location
    bool    m_has_name;         // indicates a filepath has been assigned
    string  m_filepath;         // filename of virtual image

    // header info
    bool    m_readonly;         // true if readonly
    int     m_model;            // 2200 or 3300
    vector<string> m_label;     // label text, one line at a time

    // tape state
    bool    m_dirty;            // has state, other than position, changed?
    long    m_position;         // bit offset, but there is a virtual
                                // one second leader and trailer
    int     m_length_bits;      // tape data length
    char   *m_bits;             // linear list of bit transitions
};

#endif // _INCLUDE_WANGTAPE_H_
