// This class is used to represent a virtual tape file for the Wang 3300
// emulator.  It could be useful for a utility external to its use in the
// emulator.  Read the corresponding .h for details.

#include "WangTape.h"

// =====================================================================
// the header of a virtual tape image file starts with this string:

static const int magic_len = 23;
static char magic[magic_len+1] = "Wang Virtual Cassette\x0D\x0A";

// =====================================================================
// accessory functions

static bool
read_raw(fstream &f, int bytes, char *buff)
{
    f.read(buff, bytes);
    return f.good();
}

static bool
read_raw_8b(fstream &f, uint8 *v)
{
    char buff[1];
    f.read(hdr_magic, 1);
    if (!f.good()) {
        *v = 0x00;
        return false;
    }
    *v = (uint8)buff[0];
    return true;
}

static bool
read_raw_16b(fstream &f, uint16 *v)
{
    uint8 b0, b1;
    bool ok;
    ok  = read_raw_8b(f, &b0);
    ok &= read_raw_8b(f, &b1);
    *v = (uint16)((b1 << 8) | b0);
    return ok;
}

// =====================================================================

// create new file of specified length
WangVirtualTape::WangVirtualTape(int length_kb, int model) :
    m_has_name(false),
    m_filepath(""),
    m_dirty(true)
{
    // each byte of emulated date is encoded in two bytes in memory.
    // add a factor of 2 to account for the redundant recording of each block.
    // round up.
    // FIXME: should the tape overhead (gaps) be factored in?
    m_length_bits = 2 * length_kb * 1024 * 8;

    SetModel(model);
    SetLabel("");

    // make a space for the encoded data
    m_bits = new char[(m_length_bits * 2 + 15)/16]; // 2B == 8 bits
    assert(m_bits != 0);
    Erase();

    SetOffset(0);
}


// open existing file
WangVirtualTape::WangVirtualTape(string filepath) :
    m_has_name(true),
    m_filepath(filepath),
    m_dirty(false),
    m_bits(0)
{
#if 0
FIXME: if the tape is read only, we don't need to ask for write permission.
       should we open it read only, check its status, and then close it and
       reopen it if it is r/w?  Or maybe try to open it r/w, but if it fails,
       try again as read only, and force the read only status on the tape.

       or maybe use stat()?
        #include <sys/stat.h>
        ...
        struct stat results;
        
        if (stat("input.bin", &results) == 0)
            // The size of the file in bytes is in
            // results.st_size
        else
            // An error occurred
#endif
    ifstream fs( m_filename.c_str(), ios::in | ios::binary );
    if (!fs.is_open()) {
        return; // oops -- couldn't read the file
    }

    // read the magic string
    char hdr_magic[magic_len];
    bool ok = read_raw(fs, magic_len, hdr_magic);
    if (strncmp(hdr_magic, magic, magic_len))
        ok = false;

    uint8 hdr_version;
    ok &= read_raw_8b(fs, &hdr_version);
    ok &= (hdr_version == 0);

    uint8 hdr_readonly;
    ok &= read_raw_8b(fs, &hdr_readonly);
    ok &= (hdr_readonly == 0 || hdr_readonly == 1);
    m_readonly = (hdr_readonly == 1);

    ok &= read_raw_16b(fs, &m_model);
    ok &= (m_model == 2200 || m_model == 3300);

    uint16 hdr_length;
    ok &= read_raw_16b(fs, &hdr_length);

    if (!ok) {
        fs.close();
        return;
    }

    // read it in
    char *hdr_label = new char [hdr_length+1];
    ok = read_raw(fs, hdr_length, hdr_label);
    if (!ok) {
        fs.close();
        delete[] hdr_label;
        return;
    }
    hdr_label[hdr_length] = '\0';  // c string termination

    // convert the label to a vector of strings
    m_label.clear();
    for(char *p = hdr_label,                    // current pointer
             *eob = hdr_label + hdr_length,     // end of buffer
             *sol = hdr_label;                  // start of line
        p < eob; p++) {
        // turn paired line endings into a single line ending
        bool dbl = (*p == '\x0A' && *(p+1) == '\x0D') ||
                   (*p == '\x0D' && *(p+1) == '\x0A');
        // process end of lines
        if (*p == '\x0A' || *p == '\x0D' || *p == '\0') {
            *p = '\0';
            m_label.push_back(sol);
            if (dbl) p++;
            sol = p+1;
        }
    }
    delete[] hdr_label;
    hdr_label = 0;

    // read tape image into memory
    long first_pos = fs.tellg();
    fs.seekg(0, ios::end);
    long last_pos = fs.tellg();

    long data_bytes = (last_pos - first_pos + 1) & ~1; // make sure it is even
    m_bits = new char [data_bytes];

    fs.seekg(first_pos, ios::beg);
    ok = read_raw(fs, data_bytes, m_bits);
    m_length_bits = data_bytes*8/2;     // each logical bit is encoded by two data bits
    fs.close();
    if (!ok) {
        delete[] m_bits;
        m_bits = 0;
        return;
    }

    // a newly loaded tape always starts out rewound
    SetPosition(0);
}


// destructor
WangVirtualTape::~WangVirtualTape()
{
    m_label.clear();
    delete[] m_bits;
    m_bits = 0;
}


// save any updated state; if filepath is "", then save to
// current filename, otherwise rename and save it to that file.
void
WangVirtualTape::Flush(string filepath)
{
    bool new_name = (filepath.length() > 0);
    string save_name;

    if (new_name) {
        save_name = filepath;
    } else
        save_name = m_filepath;

    // open file
    FIXME
    // write out a header
    FIXME
    // write out the data
    FIXME
}


// erase the data on the tape
void
WangVirtualTape::Erase()
{
}


// return path to file on disk
bool
WangVirtualTape::IsNamed() const
{
}


string
WangVirtualTape::Filename() const
{
}


// although it doesn't affect this library, log whether this is
// for the Wang 2200 or 3300.
int
WangVirtualTape::Model() const
{
}


void
WangVirtualTape::SetModel(int model)
{
}


// return or set the label associated with the tape.
// the label can't be longer than 64K characters.
string
WangVirtualTape::Label() const
{
}


void
WangVirtualTape::SetLabel(string label)
{
}


// a tape can be write protected, or not
bool
WangVirtualTape::IsReadOnly() const
{
}


void
WangVirtualTape::SetReadOnly(bool)
{
}


// indicates the state of the in-memory tape image may have
// changed since it was created.
bool
WangVirtualTape::IsDirty() const
{
}


// tape size, in bytes or approx minutes
int
WangVirtualTape::Length() const
{
}


int
WangVirtualTape::LengthInMinutes() const
{
}


// set read/write offset relative to start of tape, in bits
int
WangVirtualTape::Offset() const
{
}


void
WangVirtualTape::SetOffset(int)
{
}


// apparantly between data blocks
bool
WangVirtualTape::IsGap() const
{
}


// at beginning of tape?
bool
WangVirtualTape::IsBot() const
{
}


// at end of tape?
bool
WangVirtualTape::IsEot() const
{
}


// get current bit and advance
wvt_t
WangVirtualTape::ReadBit()
{
}


// write bit and advance
void
WangVirtualTape::WriteBit(wvt_t)
{
}
