// This file implements the UiTerm3315 class.
// It is the window that represents an instance of a view of the printer,
// and handles all the user interaction.

// TODO:
//
//   *) the intent is to allow the user to specify a particular offset
//      in the "Position" text box.  For now the OnXxxPosEnter() functions
//      are not hooked up by the event mapper, and the text controls are
//      set to be READONLY.

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include "UiSystem.h"
#include "UiTerm3315.h"
#include "UiTermPapertape.h"
#include "term3315.h"           // core emulation routines

#include <wx/tglbtn.h>          // for wxToggleButton

// ----------------------------------------------------------------------------
// UiTermPapertape
// ----------------------------------------------------------------------------
// This is used to draw the paper tape reader and punch controls in a subwindow
// inside the 3315 window.

enum
{
    ID_RDR_MOUNT_TXT = 1,
    ID_RDR_MOUNT,
    ID_RDR_START,
    ID_RDR_STOP,
    ID_RDR_POS_TXT,
    ID_RDR_REWIND,

    ID_PUN_MOUNT_TXT,
    ID_PUN_MOUNT,
    ID_PUN_ENABLE,
    ID_PUN_POS_TXT,
};

// dialog events to catch
BEGIN_EVENT_TABLE(UiTermPapertape, wxPanel)
    EVT_BUTTON       (ID_RDR_MOUNT,       UiTermPapertape::OnMountTape)
    EVT_BUTTON       (ID_RDR_START,       UiTermPapertape::OnRdrButton)
    EVT_BUTTON       (ID_RDR_STOP,        UiTermPapertape::OnRdrButton)
    EVT_BUTTON       (ID_RDR_REWIND,      UiTermPapertape::OnRdrButton)
//  EVT_TEXT_ENTER   (ID_RDR_POS_TXT,     UiTermPapertape::OnRdrPosEnter)
    EVT_BUTTON       (ID_PUN_MOUNT,       UiTermPapertape::OnMountTape)
    EVT_TOGGLEBUTTON (ID_PUN_ENABLE,      UiTermPapertape::OnPunButton)
//  EVT_TEXT_ENTER   (ID_PUN_POS_TXT,     UiTermPapertape::OnPunPosEnter)
END_EVENT_TABLE()


UiTermPapertape::UiTermPapertape(UiTerm3315 *parent, term3315 *term_core) :
        wxPanel(parent, wxID_ANY),
        m_parent(parent),
        m_term_core(term_core),
        m_rdr_path(0),
        m_rdr_mount_btn(0),
        m_rdr_start_btn(0),
        m_rdr_stop_btn(0),
        m_rdr_position(0),
        m_rdr_rewind_btn(0),
        m_pun_path(0),
        m_pun_mount_btn(0),
        m_pun_enable_btn(0),
        m_pun_position(0)
{
    // row 1: paper tape file selection
    wxBoxSizer *row1a = new wxBoxSizer(wxHORIZONTAL);
    row1a->Add(new wxStaticText(this, wxID_ANY, "Path"), 0, wxTop | wxALIGN_CENTER_VERTICAL, 4);
    m_rdr_path = new wxTextCtrl(this, ID_RDR_MOUNT_TXT, "<filled in later>",
                                wxDefaultPosition, wxDefaultSize,
                                wxBORDER_SUNKEN | wxTE_PROCESS_ENTER);
    row1a->Add(m_rdr_path, 1, wxLEFT | wxRIGHT | wxALIGN_CENTER_VERTICAL, 5);
    m_rdr_mount_btn = new wxButton(this, ID_RDR_MOUNT, "Browse");
    row1a->Add(m_rdr_mount_btn, 0, wxLEFT | wxALIGN_CENTER_VERTICAL, 5);

    // row 2: tape start/stop
    wxBoxSizer *row2a = new wxBoxSizer(wxHORIZONTAL);
    row2a->Add(new wxStaticText(this, wxID_ANY, "Reader"), 0, wxTop | wxALIGN_CENTER_VERTICAL, 4);
    m_rdr_start_btn = new wxButton(this, ID_RDR_START, "Start");
    m_rdr_stop_btn = new wxButton(this, ID_RDR_STOP, "Stop");
    m_rdr_rewind_btn = new wxButton(this, ID_RDR_REWIND, "Rewind");
    row2a->Add(m_rdr_start_btn,  0, wxLEFT, 5);
    row2a->Add(m_rdr_stop_btn,   0, wxLEFT, 5);
    row2a->Add(m_rdr_rewind_btn, 0, wxLEFT, 5);

    // row 3: positioning stuff
    wxBoxSizer *row3a = new wxBoxSizer(wxHORIZONTAL);
    row3a->Add(new wxStaticText(this, wxID_ANY, "Position"), 0, wxTop, 4);
    m_rdr_position = new wxTextCtrl(this, ID_RDR_POS_TXT, "(numeric)",
                                    wxDefaultPosition, wxDefaultSize,
                                    wxBORDER_SUNKEN | wxTE_PROCESS_ENTER | wxTE_READONLY);
    row3a->Add(m_rdr_position, 0, wxLEFT | wxRIGHT, 5);

    // row 1: paper tape file selection
    wxBoxSizer *row1b = new wxBoxSizer(wxHORIZONTAL);
    row1b->Add(new wxStaticText(this, wxID_ANY, "Path"), 0, wxTop | wxALIGN_CENTER_VERTICAL, 4);
    m_pun_path = new wxTextCtrl(this, ID_PUN_MOUNT_TXT, "<filled in later>",
                                wxDefaultPosition, wxDefaultSize,
                                wxBORDER_SUNKEN | wxTE_PROCESS_ENTER);
    row1b->Add(m_pun_path, 1, wxLEFT | wxRIGHT | wxALIGN_CENTER_VERTICAL, 5);
    m_pun_mount_btn = new wxButton(this, ID_PUN_MOUNT, "Browse");
    row1b->Add(m_pun_mount_btn, 0, wxLEFT | wxALIGN_CENTER_VERTICAL, 5);

    // row 2: punch enable
    wxBoxSizer *row2b = new wxBoxSizer(wxHORIZONTAL);
    row2b->Add(new wxStaticText(this, wxID_ANY, "Punch"), 0, wxTop | wxALIGN_CENTER_VERTICAL, 4);
    m_pun_enable_btn = new wxToggleButton(this, ID_PUN_ENABLE, "Enable");
    row2b->Add(m_pun_enable_btn, 0, wxLEFT, 5);

    // row 3: positioning stuff
    wxBoxSizer *row3b = new wxBoxSizer(wxHORIZONTAL);
    row3b->Add(new wxStaticText(this, wxID_ANY, "Position"), 0, wxTop, 4);
    m_pun_position = new wxTextCtrl(this, ID_PUN_POS_TXT, "(numeric)",
                                    wxDefaultPosition, wxDefaultSize,
                                    wxBORDER_SUNKEN | wxTE_PROCESS_ENTER | wxTE_READONLY);
    row3b->Add(m_pun_position, 0, wxLEFT | wxRIGHT, 5);

    // glue it together
#if 1
    wxBoxSizer *rdr_sizer = new wxStaticBoxSizer(wxVERTICAL, this, "Reader");
#else
    wxBoxSizer *rdr_sizer = new wxStaticBoxSizer(wxVERTICAL);
#endif
    rdr_sizer->Add(row1a, 1, wxALL | wxEXPAND, 5);
    rdr_sizer->Add(row2a, 1, wxALL | wxEXPAND, 5);
    rdr_sizer->Add(row3a, 1, wxALL | wxEXPAND, 5);

#if 1
    wxBoxSizer *pun_sizer = new wxStaticBoxSizer(wxVERTICAL, this, "Punch");
#else
    wxBoxSizer *pun_sizer = new wxStaticBoxSizer(wxVERTICAL);
#endif
    pun_sizer->Add(row1b, 1, wxALL | wxEXPAND, 5);
    pun_sizer->Add(row2b, 1, wxALL | wxEXPAND, 5);
    pun_sizer->Add(row3b, 1, wxALL | wxEXPAND, 5);

    wxBoxSizer *top_sizer = new wxBoxSizer(wxHORIZONTAL);
    top_sizer->Add(rdr_sizer, 1, wxEXPAND | wxALL, 5);
    top_sizer->Add(pun_sizer, 1, wxEXPAND | wxALL, 5);
    SetSizer(top_sizer);

    // refresh the papertape control state
    PapertapeUpdate(rdr_unit, pe_all);
    PapertapeUpdate(pun_unit, pe_all);
}


UiTermPapertape::~UiTermPapertape()
{
    // FIXME: anything to do?
}


// receive notification of papertape state changes
void
UiTermPapertape::PapertapeUpdate(unit_t unit, punch_event_t changed)
{
    bool tape_is_mounted = m_term_core->TapeMounted(unit);

    // update tape path/mount status
    if (changed & pe_mount) {
        // update file path
        wxString mount_label;
        wxString mount_path;
        if (tape_is_mounted) {
            mount_path  = m_term_core->GetTapePath(unit);
            mount_label = "Eject";
        } else {
            mount_path  = "-- no paper tape mounted --";
            mount_label = "Browse";
        }
        // FIXME: parameterize this better
        if (unit == rdr_unit) {
            m_rdr_path->SetEditable(!tape_is_mounted);
            m_rdr_path->SetValue(mount_path);
            m_rdr_mount_btn->SetLabel(mount_label);
            // disable controls if there is no tape in the unit
            m_rdr_start_btn->Enable(tape_is_mounted);
            m_rdr_stop_btn->Enable(tape_is_mounted);
            m_rdr_rewind_btn->Enable(tape_is_mounted);
            m_rdr_position->Enable(tape_is_mounted);
        } else {
            m_pun_path->SetEditable(!tape_is_mounted);
            m_pun_path->SetValue(mount_path);
            m_pun_mount_btn->SetLabel(mount_label);
            // disable controls if there is no tape in the unit
            m_pun_position->Enable(tape_is_mounted);
        }
    }

    // update tape offset indicator
    if (changed & (pe_mount | pe_position)) {
        // reader tape byte offset
        wxString posstr = "";
        int offset = m_term_core->TapePosition(unit);
        if (tape_is_mounted)
            posstr.Printf("%d", offset );

        if (unit == rdr_unit) {
            if (tape_is_mounted && (offset < 0))
                posstr = "<EOF>";
            m_rdr_position->SetValue(posstr);
        } else {
            m_pun_position->SetValue(posstr);
        }
    }

    // update the punch enable
    if ((unit == pun_unit) && (pe_punchen | pe_mount)) {
        m_pun_enable_btn->Enable(tape_is_mounted);
        if (!tape_is_mounted)
            m_pun_enable_btn->SetValue(false);
    }
}


void
UiTermPapertape::OnMountTape(wxCommandEvent &event)
{
    const bool reader = (event.GetId() == ID_RDR_MOUNT);
    const unit_t u     = (reader) ? rdr_unit : pun_unit;
    const int    fr_id = (reader) ? FILEREQ_PTAPE_RDR : FILEREQ_PTAPE_PUN;

    bool tape_is_mounted = m_term_core->TapeMounted(u);

    if (tape_is_mounted) {
        // we are ejecting the existing tape
        m_term_core->UnmountTape(u);
    } else {
        // browse for a file
        // open a dialog to get a filename
        wxString name;
        if (MyApp::FileReq(fr_id, "Paper Tape File (.bin)", 1, &name) != FILEREQ_OK)
            return;
        wxASSERT(!IsEmpty(name));
        m_term_core->MountTape(u, name.c_str());
    }

    // return focus to frame, otherwise the control will eat keyed input
    m_parent->Refocus();
}


void
UiTermPapertape::OnRdrPosEnter(wxCommandEvent &event)
{
    // FIXME: do something with the information
    UiAlert("Got it");
}


void
UiTermPapertape::OnPunPosEnter(wxCommandEvent &event)
{
    // FIXME: do something with the information
    UiAlert("Got it");
}


void
UiTermPapertape::OnPunButton(wxCommandEvent &event)
{
    switch (event.GetId()) {
        case ID_PUN_ENABLE:
            m_term_core->SetPunchEnabled( m_pun_enable_btn->GetValue() );
            break;
        default:
            wxFAIL;
    }

    // return focus to frame, otherwise the control will eat keyed input
    m_parent->Refocus();
}


// called when paper tape reader button control is pressed
void
UiTermPapertape::OnRdrButton(wxCommandEvent& event)
{
    switch (event.GetId()) {
        case ID_RDR_START:
            m_term_core->ReaderStart();
            break;
        case ID_RDR_STOP:
            m_term_core->ReaderStop();
            break;
        case ID_RDR_REWIND:
            m_term_core->SetTapePosition(rdr_unit, 0);
            break;
        default:
            wxFAIL;
    }

    // return focus to frame so keystrokes are still received
    m_parent->Refocus();
}


