// This file implements the UiTermPaper class.  It manages interpreting the
// stream of bytes sent to the printer, building a scrollable window of
// the printout.

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include <stdio.h>              // for saving to file
#include <ctype.h>              // for islower(), etc

#include "UiSystem.h"
#include "UiTerm3315.h"         // this module's defines
#include "term3315.h"

#define max(a,b) (((a)>(b))?(a):(b))
#define min(a,b) (((a)<(b))?(a):(b))

// setting to 1 causes input keys to be forced to upper case
// and any printed characters to be forced to upper case
#define UPPERCASE_ONLY 1

// setting to 1 prints page breaks.  The ASR33 had a continuous
// feed roll, so it isn't appropriate.
#define SHOW_PAGE_BREAKS 0

// ----------------------------------------------------------------------------
// static variables
// ----------------------------------------------------------------------------

static const int bar_h   = 3;   // height of greenbar
static const int hmargin = 3;   // number chars horizontal padding on page

// ----------------------------------------------------------------------------
// UiTermPaper
// ----------------------------------------------------------------------------

// connect the wxWindows events with the functions which process them
BEGIN_EVENT_TABLE(UiTermPaper, wxScrolledWindow)
    EVT_CHAR                    (UiTermPaper::OnChar)
    EVT_PAINT                   (UiTermPaper::OnPaint)
    EVT_ERASE_BACKGROUND        (UiTermPaper::OnEraseBackground)
    EVT_SIZE                    (UiTermPaper::OnSize)
END_EVENT_TABLE()


UiTermPaper::UiTermPaper(UiTerm3315 *parent, term3315 *term_core) :
    wxScrolledWindow(parent, -1, wxDefaultPosition, wxDefaultSize),
    m_parent(parent),
    m_term_core(term_core),
    m_scrbits(NULL),    // not assigned yet
    m_charcell_w(0),
    m_charcell_h(0),
    m_greenbar(false),
    m_linelength(72),   // default (ASR-33 limit)
    m_pagelength(66)    // default
{
    // the means don't perform a screen-to-screen blit to effect
    // the scroll.  just let the app redraw everything.
    EnableScrolling(false, false);

    PrintClear();
}

// free resources on destruction
UiTermPaper::~UiTermPaper()
{
    m_printstream.Clear();

    if (m_scrbits != NULL) {
        delete m_scrbits;
        m_scrbits = NULL;
    }
}

// ---- public methods ----

// set the point size of the text
void
UiTermPaper::SetFontSize(const int size)
{
    wxClientDC dc(this);

#if 1
    m_font = wxFont(size, wxMODERN, wxNORMAL, wxNORMAL, FALSE);
#else
//  m_font = wxFont(size, wxMODERN, wxNORMAL, wxNORMAL, FALSE, "Teleprinter");
    m_font = wxFont(size, wxMODERN, wxNORMAL, wxNORMAL, FALSE, "Kingthings Trypewriter");
#endif
    m_font.SetNoAntiAliasing();
    dc.SetFont( m_font );

    m_fontsize   = size;
    m_charcell_w = dc.GetCharWidth();
    m_charcell_h = dc.GetCharHeight();

    //set the number of rows in the view
    m_chars_h = m_scrpix_h / m_charcell_h;

    //set the number of columns in the view (fixed pitch font makes this easy)
    m_chars_w = m_scrpix_w/m_charcell_w;

    UpdateView();
}

void
UiTermPaper::GetFontSize(int &size)
{
    size = m_fontsize;
}

void
UiTermPaper::SetPageAttributes(int linelength, int pagelength)
{
    m_linelength = linelength;
    m_pagelength = pagelength;
    UpdateView();
}


void
UiTermPaper::GetPageAttributes(int &linelength, int &pagelength)
{
    linelength = m_linelength;
    pagelength = m_pagelength;
}

void
UiTermPaper::GetCellAttributes(int *cell_w, int *cell_h)
{
    *cell_w = m_charcell_w;
    *cell_h = m_charcell_h;
}

// emit a character to the display
void
UiTermPaper::OutChar(uint8 byte)
{
    byte &= 0x7F;

#if UPPERCASE_ONLY
    // convert lower case to upper case
    if (islower(byte))
        byte = toupper(byte);
#endif

    switch (byte) {

/*
The special Control Codes for the UiTermPaper are:
Function Hex Code
Description

ALARM HEX (07)
Generates an audible tone about two seconds in
duration in the speaker at the rear of the printer.

VERTICAL TAB HEX(0B)
Advances paper until the next hole in channel 5 of
the Vertical Format Unit paper tape is reached.

ELONGATED HEX(0E)
CHARACTER
Prints a line up to 66 characters as expanded
(double-width) characters. (See Section III.)

DELETE HEX (7F)
Clears buffer of characters sent before the '7F'

Some of the Wang printers were modified selectrics and had a different
set of control codes. For instance, you could set and clear tabstops
just like a real selectric so that printing CTRL-I (tab) would advance
to the next tab stop.
*/

        case 0x09:
            // HORIZONTAL TAB HEX(09)
            // Assumed to be hardcoded to tabstop of 8 characters
            {
                int last_line = m_printstream.Count();
                int len = m_printstream[last_line-1].Len();
                do {
                    if (len >= m_linelength)
                        break;
                    m_printstream[last_line-1] += ' ';
                    len++;
                } while (len%8 != 0);
            }
            break;

        case 0x0A:
            // LINE FEED HEX(0A)
#if 0
            // Advance the paper one line, but don't change column
            {
                int last_line = m_printstream.Count();
                int len = m_printstream[last_line-1].Len();
                EmitLine();
                for(int n=0; n < len; n++)
                    m_printstream[last_line] += ' ';
            }
#else
            // treat it like carriage return, unless we are in column 0.
            // CR and CR/LF are the same thing.  LF/CR causes two lines.
            {
                int last_line = m_printstream.Count();
                int len = m_printstream[last_line-1].Len();
                if (len > 0)
                    EmitLine();
            }
#endif
            break;

        case 0x0C:
            // FORM FEED HEX(0C)
            // Advances paper until the next hole in channel 7
            // of the Vertical Format Unit paper tape is reached.

            // Emit line to the printer, add empty lines to advance stream to the next page.
            {
                int last_line = m_printstream.Count();
                int len = m_printstream[last_line-1].Len();
                if (len > 0)
                    EmitLine();
                FormFeed();
            }
            break;

        case 0x0D:
            // CARRIAGE RETURN HEX(0D)
            // Causes the line of characters stored in the printer
            // buffer to be printed. An automatic line feed
            // occurs after the line has been printed and the
            // print head returns to the left side of the printer
            // carrier.

            // Emit the line to the print stream.
            EmitLine();

            break;

        default:        // just a character
            {
                int last_line = m_printstream.Count();
                int len = m_printstream[last_line-1].Len();
                if ((byte >= 32) && (len < m_linelength)) {
                    // accumulate the partial line
                    m_printstream[last_line-1] += static_cast<char>(byte);
                    UpdateView();
                }
            }
            break;
    }
}


// save the contents of the printer to a file
void
UiTermPaper::SaveToFile()
{
    wxString fullpath;
    int r = MyApp::FileReq(FILEREQ_PRINTER, "Save printer log file", 0, &fullpath);
    if (r == FILEREQ_OK) {

        // save the file
        const char *filename = fullpath.c_str();
        FILE *fd = fopen(filename, "wb");       // only need to write
        if (fd == NULL) {
            UiAlert("Couldn't open '%s' for writing", filename);
            return;
        }

        int max = m_printstream.GetCount();

        for(int n = 0; n < max; n++) {
#ifdef __WXMSW__
            wxString tmpline(m_printstream[n] + "\r\n");
#else
            wxString tmpline(m_printstream[n] + "\n");
#endif
            int len = tmpline.Length();
            int stat = fwrite(tmpline.c_str(), len, 1, fd);
            if (stat != 1) {
                UiAlert("Error writing to line %d of '%s'", n+1, filename);
                fclose(fd);
                return;
            }
        }
        fclose(fd);
    }
}


// clear the printer contents
void
UiTermPaper::PrintClear()
{
    m_printstream.Clear();      // log of all complete lines
    m_printstream.Add("");      // start of first line

    ScrollbarSet(0, 0, false);
    InvalidateAll();
}

// return the number of pages in the current copy of the printstream
int
UiTermPaper::NumberOfPages()
{
    int num_rows = m_printstream.GetCount();
    return ((num_rows + m_pagelength-1) / m_pagelength);  // round up
}

void
UiTermPaper::ScrollbarSet(int xpos, int ypos, bool redraw)
{
    SetScrollbars(  m_charcell_w,               // pixels per scroll unit x
                    m_charcell_h,               // pixels per scroll unit y
                    m_linelength + 2*hmargin,   // number of units x
                    m_printstream.GetCount(),   // number of units y,
                    xpos,                       // x position in scroll units
                    ypos,                       // y position in scroll units
                    !redraw);                   // redraw the screen
}

// ---- private methods ----

// route keystrokes to the core terminal keyboard routines
void
UiTermPaper::OnChar(wxKeyEvent &event)
{
    int kc = event.GetKeyCode();

#if UPPERCASE_ONLY
    // convert lower case to upper case
    if (islower(kc))
        kc = toupper(kc);
#endif

    TermCore()->Keystroke(kc);
}


void
UiTermPaper::OnPaint(wxPaintEvent &WXUNUSED(event))
{
    // have we scrolled?
    int firstCol, firstLine;
    GetViewStart(&firstCol, &firstLine);

    // scroll-wheeling up can produce negative offsets
    firstLine = max(firstLine, 0);
    firstCol  = max(firstCol,  0);

    // scroll-wheeling up can produce large offset
    int num_rows = m_printstream.GetCount();    // # rows in log
    if (num_rows < m_chars_h)
        firstLine = 0;

    wxPaintDC dc(this);
    DrawScreen(dc, firstCol, firstLine);
    UpdateStatusbar();
}

void
UiTermPaper::OnSize(wxSizeEvent& event)
{
    int width, height;
    GetClientSize(&width, &height);

    m_scrpix_w = width;
    m_scrpix_h = height;

    //reset the number of rows in the view
    if (m_charcell_h != 0)              // the first time through, when font has not initialized, it is zero
        m_chars_h = height / m_charcell_h;
    if (m_charcell_w != 0)
        m_chars_w = width / m_charcell_w;

    m_scrpix_h = m_scrpix_h + m_charcell_h; //add one extra row to make scrolling work

    if (m_scrbits != NULL)
        delete m_scrbits;
    m_scrbits = new wxBitmap(width,height);     // same depth as display

    UpdateView();

    event.Skip();       // do the rest of the OnSize processing
}

// intercept this event to prevent flashing as the system so helpfully
// clears the window for us before generating the OnPaint event.
void
UiTermPaper::OnEraseBackground(wxEraseEvent &WXUNUSED(event))
{
    // do nothing
}

// refresh the entire window
void
UiTermPaper::DrawScreen(wxDC& dc, int startCol, int startRow)
{
    // update the view image bitmap
    GenerateScreen(startCol, startRow);

    // make a DC to hang the bitmap onto
    wxMemoryDC imgDC;
    imgDC.SelectObject(*m_scrbits);

    // blast the image to the screen
    dc.Blit(0, 0,                       // dest
            m_scrpix_w, m_scrpix_h,     // size
            &imgDC, 0, 0);              // source
}

// update the pixmap of the screen image
// if greenbar mode is on, the layout is like this:
//    |<three chars of white>|<N chars of green or white>|<three chars of white>|
//    the page alternates three lines of white, three of green, ...
void
UiTermPaper::GenerateScreen(int startCol, int startRow)
{
    // width of virtual paper, in pixels
    const int page_w = m_charcell_w * (m_linelength + 2*hmargin);

    // amount of background to the left/right of virtual page in viewport
    const int left_bg_w  = max(0, (m_scrpix_w - page_w)/2);
    const int right_bg_w = max(0, m_scrpix_w - left_bg_w);

    // left edge of paper relative to the viewport (can be negative)
    const int left_edge = (-startCol * m_charcell_w)    // if scrolled left
                        + left_bg_w;                    // if viewport > page_w

    // right edge of paper relative to the viewport, exclusive
    const int right_edge = left_edge + page_w - 1;

    // this is the number of characters to skip at the start of each row
    const int skip_chars = max((startCol - hmargin), 0);

    // if the viewport is wider than the page, there shouldn't be
    // any startCol offset
    wxASSERT( (startCol == 0) || (left_edge < 0) );

    if (m_scrbits == NULL) {
        wxASSERT_MSG( (m_scrbits != NULL), "m_scrbits is null in printer");
        return;
    }

    wxMemoryDC imgDC;
    imgDC.SelectObject(*m_scrbits);             // the thing we're drawing on

    // draw page background to white
    {
        if (m_greenbar) {
            imgDC.SetPen(*wxWHITE_PEN);
            imgDC.SetBrush(*wxWHITE_BRUSH);
        } else {
            wxColor bg = wxColour(0xFC,0xF8,0xD8);  // slightly yellow tint
            imgDC.SetPen(bg);
            imgDC.SetBrush(bg);
        }
        imgDC.DrawRectangle( left_edge, 0,              // origin
                             page_w, m_scrpix_h );      // width, height
    }

    // draw greyed out region on the right and left
    // (right_bg_w >= left_bg_w) so only need to check right_bg_w
    if (right_bg_w >= 0) {
        imgDC.SetPen(*wxGREY_PEN);
        imgDC.SetBrush(*wxGREY_BRUSH);
        imgDC.DrawRectangle( 0, 0,              // origin
                             left_bg_w,         // width
                             m_scrpix_h );      // height
        imgDC.DrawRectangle( right_edge, 0,     // origin
                             right_bg_w,        // width
                             m_scrpix_h );      // height

        // draw black edge to paper for emphasis
        imgDC.SetPen(*wxBLACK_PEN);
        imgDC.DrawLine( left_edge-1, 0,         // origin
                        left_edge-1,            // width
                        m_scrpix_h);            // height
        imgDC.DrawLine( right_edge-1, 0,        // origin
                        right_edge-1,           // width
                        m_scrpix_h);            // height

        imgDC.SetPen(wxNullPen);
        imgDC.SetBrush(wxNullBrush);
    }

    // if greenbar mode, draw green rounded rectangles
    if (m_greenbar) {
        wxColor lightGreen = wxColour(0xB0,0xFF,0xB0);
        wxColor darkGreen  = wxColour(0x00,0x80,0x00);
        wxBrush rectfill(lightGreen);
        wxPen   rectoutline(darkGreen);
        imgDC.SetPen(rectoutline);
        imgDC.SetBrush(rectfill);

        int bar_2h = bar_h*2;   // twice height of greenbar
        int first_greenbar = ((startRow                )/bar_2h)*bar_2h + bar_h;
        int last_greenbar  = ((startRow + m_chars_h + 1)/bar_2h)*bar_2h + bar_h;

        for(int bar = first_greenbar; bar <= last_greenbar; bar += bar_2h) {
            int yoff = (bar - startRow) * m_charcell_h;
            int xoff = left_edge + hmargin* m_charcell_w - m_charcell_w/2;  //  \ expand it 1/2 char
            int width  = m_linelength * m_charcell_w     + m_charcell_w;    //  / on each side
            int height = bar_h * m_charcell_h;
            double radius = m_charcell_w * 0.5;
            imgDC.DrawRoundedRectangle(xoff,yoff, width,height, radius);
        }

        imgDC.SetPen(wxNullPen);
        imgDC.SetBrush(wxNullBrush);
    }

    // draw page breaks
    if (SHOW_PAGE_BREAKS) {
        wxColor gray = wxColour(0x80, 0x80, 0x80);
        wxPen breakpen(gray, 1, wxUSER_DASH);
        wxDash dashArray[] = { 2, 5 };  // pixels on, pixels off
        breakpen.SetDashes(2, dashArray);
        imgDC.SetPen(breakpen);

        int first_break = ((startRow                )/m_pagelength)*m_pagelength;
        int last_break  = ((startRow + m_chars_h + 1)/m_pagelength)*m_pagelength;
        if (startRow == 0)
            first_break += m_pagelength;        // skip first break
        for(int brk = first_break; brk <= last_break; brk += m_pagelength) {
            int x_off = left_edge;
            int y_off = m_charcell_h * (brk - startRow);
            int x_end = left_edge + page_w;
            imgDC.DrawLine( x_off, y_off,       // from (x,y)
                            x_end, y_off);      // to   (x,y)
        }

        imgDC.SetPen(wxNullPen);
    }

    // draw each row of the text
    {
        imgDC.SetBackgroundMode(wxTRANSPARENT); // in case of greenbar mode
        imgDC.SetTextBackground(*wxWHITE);      // moot if greenbar mode
        imgDC.SetTextForeground(*wxBLACK);      // always
        imgDC.SetFont(m_font);

        int num_rows = m_printstream.GetCount();
        wxString line;

        for(int row = 0; row < m_chars_h + 1; row++) {

            if (startRow + row < num_rows) {
                // the line exists
                line = m_printstream[startRow + row];
                int line_len = line.Len();
                size_t nchars = (m_linelength+hmargin) - skip_chars;
                if ((skip_chars > 0) || (int(nchars) < line_len)) {
                    // chop off any chars to the left of the display or
                    // to the right of the right edge of the virtual paper
                    line = line.Mid(skip_chars, nchars);
                }
                int x_off = left_bg_w + m_charcell_w * max(hmargin - startCol, 0);
                int y_off = m_charcell_h * row;
                imgDC.DrawText(line, x_off, y_off);

                // draw a phantom box showing where the print head is
                if (startRow+row == num_rows-1) {
                    wxPen rectoutline(wxColour(0xA0,0xA0,0xA0));
                    imgDC.SetBrush(*wxTRANSPARENT_BRUSH);
                    imgDC.SetPen(rectoutline);
                    int x_off = left_bg_w + m_charcell_w * max(hmargin - startCol, 0)
                                          + m_charcell_w * line.Len();
                    int y_off = m_charcell_h * row + 1;
                    // draw it if it isn't scrolled off to the left
                    if (hmargin + line_len >= startCol)
                        imgDC.DrawRectangle(x_off, y_off, m_charcell_w-2, m_charcell_h-2);
                    imgDC.SetPen(wxNullPen);
                }
            }

        } // for(row)
    }
}

// emit current buffer to the print stream
void
UiTermPaper::EmitLine()
{
    m_printstream.Add(""); // add a blank line

    UpdateView();
}

// emit lines to simulate a form feed
void
UiTermPaper::FormFeed()
{
    int linesToAdd = m_pagelength - (m_printstream.GetCount() % m_pagelength);
    wxString buffer = "";

    for(int i = 0; i < linesToAdd; i++) {
        // call emit line to that current line buffer is flushed
        // and to make sure page oriented functions such as "print as you go" are invoked
        // if necessary
        EmitLine();
    }
    UpdateView();
}

// check redraw of screen, set scrollbars, update statusbar text
void
UiTermPaper::UpdateView()
{
    // we need to determine here if the screen needs to be redrawn.
    // if this line number in the stream is within m_chars_h of the
    // first visible line, we need to redraw.  otherwise not.
    int first_visible_row, first_visible_char;
    GetViewStart(&first_visible_char, &first_visible_row);
    const int endrow = first_visible_row + m_chars_h;   // last row on screen
    const int num_rows = m_printstream.GetCount();      // # rows in log

    if (num_rows <= m_chars_h) {
        // the entire print state fits on screen
        ScrollbarSet(0, 0, true);
    } else if (num_rows == endrow+1) {
        // if the new row is one off the end,
        // scroll to make sure the new row is still visible
        ScrollbarSet(first_visible_char, first_visible_row+1, true);
    } else {
        // the newly added row is off the portion we are looking at.
        // call ScrollbarSet to make sure the scrollbar is updated,
        // but keep the top row unchanged
        ScrollbarSet(first_visible_char, first_visible_row, true);
    }

    UpdateStatusbar();
    InvalidateAll();
}

// update the statusbar text
void
UiTermPaper::UpdateStatusbar()
{
#if __WXMAC__
    // on the mac, the code in here causes a paint refresh, which calls this again and we get in an infinite loop
    return;
#endif
    int first_visible_row, first_visible_char;
    GetViewStart(&first_visible_char, &first_visible_row);
    const int num_rows = m_printstream.GetCount();      // # rows in log

    // update statusbar text
    // the current line/page reported are based on the last visible
    // on the page.  it gets a little weird in that the last visible
    // line is ambiguous in the case of partial rows.
    const int last_line = min(first_visible_row + m_chars_h+1, num_rows);
    wxString msg;
#if SHOW_PAGE_BREAKS
    const int cur_page = (last_line + m_pagelength-1) / m_pagelength;
    const int num_pages = NumberOfPages();
    msg.Printf("Page %d of %d (line %d of %d)",
                cur_page, num_pages, last_line, num_rows);
#else
    msg.Printf("Line %d of %d", last_line, num_rows);
#endif
    m_parent->SetStatusText(msg);
}
