// This file implements the UiTerm3315 class.
// It is the window that represents an instance of a view of the printer,
// and handles all the user interaction.

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include "UiSystem.h"
#include "UiTerm3315.h"
#include "UiTermPapertape.h"
#include "term3315.h"           // core emulation routines

// ----------------------------------------------------------------------------
// constants
// ----------------------------------------------------------------------------

// IDs for the controls and the menu commands
enum
{
    // menu items
    File_Exit = wxID_EXIT,
    File_SaveAs = wxID_SAVEAS,
    File_PrintClear = 1,

    Display_Papertape,

    Display_FontSize8,
    Display_FontSize10,
    Display_FontSize12,
    Display_FontSize14,
    Display_FontSize18,
};

// ----------------------------------------------------------------------------
// UiTerm3315
// ----------------------------------------------------------------------------

// connect the wxWindows events with the functions which process them
BEGIN_EVENT_TABLE(UiTerm3315, wxFrame)

    EVT_MENU      (File_SaveAs,           UiTerm3315::OnFileSaveAs)
    EVT_MENU      (File_PrintClear,       UiTerm3315::OnFilePrintClear)
    EVT_MENU      (File_Exit,             UiTerm3315::OnFileExit)

    EVT_MENU      (Display_Papertape,     UiTerm3315::OnDisplayPapertape)
    EVT_MENU      (Display_FontSize8,     UiTerm3315::OnFontSize)
    EVT_MENU      (Display_FontSize10,    UiTerm3315::OnFontSize)
    EVT_MENU      (Display_FontSize12,    UiTerm3315::OnFontSize)
    EVT_MENU      (Display_FontSize14,    UiTerm3315::OnFontSize)
    EVT_MENU      (Display_FontSize18,    UiTerm3315::OnFontSize)

    // non-menu event handlers
    EVT_MENU_OPEN (UiTerm3315::OnMenuOpen)
    EVT_MENU_CLOSE(UiTerm3315::OnMenuClose)
    EVT_CLOSE     (UiTerm3315::OnClose)

    // help menu items do whatever they need to do
    HELP_MENU_EVENT_MAPPINGS()

END_EVENT_TABLE()


// constructor
UiTerm3315::UiTerm3315(const int term_number,
                       term3315 *term_core) :
        wxFrame((wxFrame *)NULL, -1, "", wxDefaultPosition, wxDefaultSize,
                 wxDEFAULT_FRAME_STYLE | wxNO_FULL_REPAINT_ON_RESIZE),
        m_term_core(term_core),
        m_term_number(term_number),
        m_menuBar(0),
        m_statusBar(0),
        m_termpaper(0),
        m_papertape(0)
{
#ifndef __WXMAC__
    // set the frame icon
    SetIcon(wxICON(wang));
#endif
    wxString title;
    title.Printf("Wang 3315 Terminal %d", term_number+1);
    this->SetTitle(title);

    MakeMenubar();      // create menubar

    m_statusBar = new wxStatusBar(this);
    SetStatusBar(m_statusBar);

    m_termpaper = new UiTermPaper(this, term_core);

    m_papertape = new UiTermPapertape(this, term_core);
    m_papertape->Show(false);  // hidden by default

    m_sizer = new wxBoxSizer(wxVERTICAL);
    m_sizer->Add(m_papertape, 0, wxEXPAND);
    m_sizer->Add(m_termpaper, 1, wxEXPAND);
    SetSizer(m_sizer);
 
    GetDefaults();      // get configuration options, or supply defaults
    SetMenuChecks();

    Show(true);
}


// destructor
UiTerm3315::~UiTerm3315()
{
    delete m_papertape;
    delete m_termpaper;
}


// create menubar
void
UiTerm3315::MakeMenubar()
{
    wxMenu *menuFile = new wxMenu;
    menuFile->Append(File_SaveAs,        "Save &As",         "Save the contents of the printer to a file");
    menuFile->Append(File_PrintClear,    "&Clear printer",   "Clear the printer");
    menuFile->Append(File_Exit,          "E&xit",            "Exit the program");

    wxMenu *menuDisplay = new wxMenu;
    menuDisplay->Append(Display_Papertape,  "Show &Papertape", "Show/hide papertape controls", wxITEM_CHECK);
    menuDisplay->AppendSeparator();
    menuDisplay->Append(Display_FontSize8,  "Font Size  8", "Set display font to  8 point", wxITEM_CHECK);
    menuDisplay->Append(Display_FontSize10, "Font Size 10", "Set display font to 10 point", wxITEM_CHECK);
    menuDisplay->Append(Display_FontSize12, "Font Size 12", "Set display font to 12 point", wxITEM_CHECK);
    menuDisplay->Append(Display_FontSize14, "Font Size 14", "Set display font to 14 point", wxITEM_CHECK);
    menuDisplay->Append(Display_FontSize18, "Font Size 18", "Set display font to 18 point", wxITEM_CHECK);
 
    // make the help menu (as if it isn't obvious below!)
    wxMenu *menuHelp = MyApp::MakeHelpMenu();

    // now append the freshly created menu to the menu bar...
    m_menuBar = new wxMenuBar;
    m_menuBar->Append(menuFile,    "&File");
    m_menuBar->Append(menuDisplay, "&Display");
    m_menuBar->Append(menuHelp,    "&Help");

    // ... and attach this menu bar to the frame
    SetMenuBar(m_menuBar);
}


// this is called just before a menu is displayed.
// set the check status for each of the menu items.
void
UiTerm3315::SetMenuChecks()
{
    // ----- display -----------------------------------
    m_menuBar->Check(Display_Papertape,  m_papertape->IsShown());

    int fontsize;
    m_termpaper->GetFontSize(fontsize);

    m_menuBar->Check(Display_FontSize8,  fontsize ==  8);
    m_menuBar->Check(Display_FontSize10, fontsize == 10);
    m_menuBar->Check(Display_FontSize12, fontsize == 12);
    m_menuBar->Check(Display_FontSize14, fontsize == 14);
    m_menuBar->Check(Display_FontSize18, fontsize == 18);
}


// save Term options to the config file
void
UiTerm3315::SaveDefaults()
{
    wxString subgroup;
    subgroup.Printf("ui/Term-%02x", m_term_number);

    // save position and size
    ConfigWriteWinGeom(this, subgroup);

    int plen, llen, fontsize;
    m_termpaper->GetPageAttributes(llen, plen);
    m_termpaper->GetFontSize(fontsize);

    // save page attributes
    ConfigWriteInt(subgroup, "pagelength",   plen);
    ConfigWriteInt(subgroup, "linelength",   llen);
    ConfigWriteInt(subgroup, "fontsize",     fontsize);
}


// get Term options from the config file, supplying reasonable defaults
void
UiTerm3315::GetDefaults()
{
    wxString valstr;
    int v;
    bool b;

    // pick up screen color scheme
    wxString subgroup;
    subgroup.Printf("ui/Term-%02x", m_term_number);

    // pick up screen location and size
    wxRect default_geom(50,50,690,380);
    ConfigReadWinGeom(this, subgroup, &default_geom);

    // pick up screen font size
    b = ConfigReadInt(subgroup, "fontsize", &v);
    int fontsize = (b && (v >= 8) && (v <= 28)) ? v : 10;
    m_termpaper->SetFontSize(fontsize);

    // pick up page attributes
    int plen, llen;
    (void)ConfigReadInt(subgroup, "pagelength", &plen, 66);
    (void)ConfigReadInt(subgroup, "linelength", &llen, 72);  // standard for ASR-33
    m_termpaper->SetPageAttributes(llen, plen);
}

// called to request a window to close (deferred)
// this overrides the superclass so that we can always write out
// the frame's options to the config file, whether the frame is
// closing due to application exit, or just a reconfiguration.
bool
UiTerm3315::Destroy()
{
    SaveDefaults();             // save config options
    return wxFrame::Destroy();  // let superclass handle real work
}

// event handlers

// called when File/Close is selected
void
UiTerm3315::OnFileExit(wxCommandEvent& WXUNUSED(event))
{
    Close();
}

// save the contents of the virtual printer
void
UiTerm3315::OnFileSaveAs(wxCommandEvent& WXUNUSED(event))
{
    m_termpaper->SaveToFile();
}


// erase the contents of the printer log
void
UiTerm3315::OnFilePrintClear(wxCommandEvent& WXUNUSED(event))
{
    m_termpaper->PrintClear();
}


// show/hide the papertape control panel
void
UiTerm3315::OnDisplayPapertape(wxCommandEvent& WXUNUSED(event))
{
    const bool showing = m_papertape->IsShown();
    wxSizer *top_sizer = GetSizer();
    top_sizer->Show(m_papertape, !showing);
    Layout();

    if (m_papertape->IsShown()) {
        // the papertape controls are showing.  make sure the minimum
        // size is respected so that the window doesn't cause all the
        // controls to step on each other.
        wxSize minsize = top_sizer->GetMinSize();
        this->SetMinSize(minsize);
        // if the window is already narrower than the minimum,
        // force the window to grow wider
        wxSize cursize = top_sizer->GetSize();
        if (cursize.GetWidth() < minsize.GetWidth())
            this->SetClientSize(minsize.GetWidth(),cursize.GetHeight());
    } else {
        // papertape control isn't showing -- let user make it small as they want
        this->SetMinSize(wxDefaultSize);
    }
}


void
UiTerm3315::OnFontSize(wxCommandEvent& event)
{
    int size;
    switch (event.GetId()) {
        case Display_FontSize8:  size =  8; break;
        case Display_FontSize10: size = 10; break;
        case Display_FontSize12: size = 12; break;
        case Display_FontSize14: size = 14; break;
        case Display_FontSize18: size = 18; break;
        default:
            // the user must have edited the ini file and added a bogus
            // font size specifier.  default to something sane.
            size = 10;
            break;
    }
    m_termpaper->SetFontSize(size);
}


// called when the window is manually closed ("X" button, or sys menu)
void
UiTerm3315::OnClose(wxCloseEvent& WXUNUSED(event))
{
    MyApp::Terminate();
}


void
UiTerm3315::OnMenuOpen(wxMenuEvent& WXUNUSED(event))
{
    // for 1-of-N selection choices, indicate which is currently selected
    SetMenuChecks();
}


void
UiTerm3315::OnMenuClose(wxMenuEvent& WXUNUSED(event))
{
    // the menu description writes to the statusbar.
    // forcing a Refresh() causes the normal status line to be restored.
    Refresh();
}


// other class functions ...

// emit a character to the display
void
UiTerm3315::OutChar(uint8 byte)
{
    // send it to the Term
    m_termpaper->OutChar(byte);
}

void
UiTerm3315::SetStatusText(const wxString &msg)
{
    m_statusBar->SetStatusText(msg);
}

// receive notification of papertape state changes
void
UiTerm3315::PapertapeUpdate(unit_t unit, punch_event_t changed)
{
    m_papertape->PapertapeUpdate(unit, changed);
}
