// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include "system3300.h"
#include "UiSystem.h"
#include "UiSystemConfigDlg.h"

enum
{
    ID_MEMSIZE_CHOICE = 1,
    ID_SPEED_REGULATED,
    ID_NUM_TERMINALS,
    ID_NUM_CASSETTE_DRIVES,

    ID_BTN_REVERT = 300
};


// dialog events to catch
BEGIN_EVENT_TABLE(SystemConfigDlg, wxDialog)

    EVT_CHOICE(ID_MEMSIZE_CHOICE,       SystemConfigDlg::OnMemsizeChoice)
    EVT_CHECKBOX(ID_SPEED_REGULATED,    SystemConfigDlg::OnSpeedRegulated)
    EVT_CHOICE(ID_NUM_TERMINALS,        SystemConfigDlg::OnNumTerminals)
    EVT_CHOICE(ID_NUM_CASSETTE_DRIVES,  SystemConfigDlg::OnNumCassetteDrives)
    EVT_BUTTON(wxID_ANY,                SystemConfigDlg::OnButton)
END_EVENT_TABLE()


SystemConfigDlg::SystemConfigDlg(wxFrame *parent)
        : wxDialog(parent, -1, "System Configuration",
                   wxDefaultPosition, wxDefaultSize,
                   wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER)
{
    const int h_text_margin = 8;

    hSpeedRegulated = new wxCheckBox(this, ID_SPEED_REGULATED,
                                           "Regulated CPU Speed");

    {
        hMemsize = new wxChoice(this, ID_MEMSIZE_CHOICE);
        static const int typical_ram_sizes[] =
            { 4, 8, 12, 16, 20, 24, 28, 32, 40, 48, 56, 64 };
        for(int n=0; n < sizeof(typical_ram_sizes)/sizeof(int); n++) {
            wxString label;
            label.Printf("%2d KB", typical_ram_sizes[n]);
            hMemsize->Append(label, (void*)typical_ram_sizes[n]);
        }
    }

    {
        hNumTerms = new wxChoice(this, ID_NUM_TERMINALS);
        for(int n=1; n <= 8; n++) {
            wxString label;
            label.Printf("%1d", n);
            hNumTerms->Append(label, (void*) n);
        }
    }

    {
        hNumCassettes = new wxChoice(this, ID_NUM_CASSETTE_DRIVES);
        for(int n=0; n <= 8; n += 2) {
            wxString label;
            label.Printf("%1d", n);
            hNumCassettes->Append(label, (void*) n);
        }
    }

    wxFlexGridSizer *grid = new wxFlexGridSizer(3, 2, 2, 0);

    grid->Add(new wxStaticText(this, -1, "RAM"), 0,
                wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL | wxRIGHT, h_text_margin);
    grid->Add(hMemsize, 0);

    grid->Add(new wxStaticText(this, -1, "Terminals"), 0,
                wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL | wxRIGHT, h_text_margin);
    grid->Add(hNumTerms, 0);

    grid->Add(new wxStaticText(this, -1, "Cassette Drives"), 0,
                wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL | wxRIGHT, h_text_margin);
    grid->Add(hNumCassettes, 0);

    wxBoxSizer *ctlsizer = new wxBoxSizer(wxVERTICAL);
    ctlsizer->Add(hSpeedRegulated, 0, wxALL, 5);
    ctlsizer->Add(grid, 0, wxALL, 5);

#if 0
    // and this option
    hWarnIO = new wxCheckBox(this, ID_CHK_WARN_IO,
                                   "Warn on Invalid IO Device Access");
#endif

    // put two buttons side by side
    hBtnRevert = new wxButton(this, ID_BTN_REVERT, "Revert");
    hBtnOK     = new wxButton(this, wxID_OK,       "OK");
    hBtnCancel = new wxButton(this, wxID_CANCEL,   "Cancel");

    wxBoxSizer *button_sizer = new wxBoxSizer( wxHORIZONTAL );
    button_sizer->Add(hBtnRevert, 0, wxALL, 10);
    button_sizer->Add(hBtnOK,     0, wxALL, 10);
    button_sizer->Add(hBtnCancel, 0, wxALL, 10);
#ifdef __WXMAC__
    // the cancel button was running into the window resizing grip
    button_sizer->AddSpacer(10);
#endif
    hBtnRevert->Disable();      // until something changes
  
    // config grids on top, confirmation buttons on the bottom
    wxBoxSizer *topsizer = new wxBoxSizer( wxVERTICAL );
    topsizer->Add(ctlsizer,     1, wxEXPAND);           // vertically stretchable
    topsizer->Add(button_sizer, 0, wxALIGN_RIGHT);      // vertically unstretchable

    m_config = new SystemConfig(*System3300::GetConfig());  // start with current state
    UpdateDlg();        // select current options

    // tell the thing to get to work
    SetSizer(topsizer);                 // use the sizer for layout
    topsizer->SetSizeHints(this);       // set size hints to honor minimum size

    GetDefaults();              // get default size & location
}


// update the display to reflect the current state
void
SystemConfigDlg::UpdateDlg()
{
    int regulated = m_config->GetCpuSpeedRegulated();
    hSpeedRegulated->SetValue(regulated ? 1 : 0);

    int ram_kb = m_config->GetRamKB();
    int n = (ram_kb <= 32) ? (ram_kb/4 - 1) : (ram_kb/8 + 3);
    hMemsize->SetSelection(n);

    int terms = m_config->GetNumTerminals();
    hNumTerms->SetSelection(terms - 1);

    int cassettes = m_config->GetNumCassetteDrives();
    hNumCassettes->SetSelection(cassettes/2);

#if 0
    int warnio = iomgr_get_warnio();
    hWarnIO->SetValue(warnio == 1);
#endif

    // FIXME: a cleaner way would be to implement an operator== member to config class
    const SystemConfig old_cfg(*System3300::GetConfig());
    bool changed = (old_cfg.GetCpuSpeedRegulated() != m_config->GetCpuSpeedRegulated())
                || (old_cfg.GetRamKB()             != m_config->GetRamKB())
                || (old_cfg.GetNumTerminals()      != m_config->GetNumTerminals())
                || (old_cfg.GetNumCassetteDrives() != m_config->GetNumCassetteDrives());
    hBtnRevert->Enable(changed);
}


void
SystemConfigDlg::OnSpeedRegulated( wxCommandEvent& WXUNUSED(event) )
{
    bool checked = hSpeedRegulated->IsChecked();
    m_config->SetCpuSpeedRegulated(checked);
    UpdateDlg();
}


void
SystemConfigDlg::OnMemsizeChoice( wxCommandEvent& WXUNUSED(event) )
{
    int selection = hMemsize->GetSelection();
    m_config->SetRamKB(int(hMemsize->GetClientData(selection)));
    UpdateDlg();
}


void
SystemConfigDlg::OnNumTerminals( wxCommandEvent& WXUNUSED(event) )
{
    int selection = hNumTerms->GetSelection();
    m_config->SetNumTerminals(int(hNumTerms->GetClientData(selection)));
    UpdateDlg();
}

void
SystemConfigDlg::OnNumCassetteDrives( wxCommandEvent& WXUNUSED(event) )
{
    int selection = hNumCassettes->GetSelection();
    m_config->SetNumCassetteDrives(int(hNumCassettes->GetClientData(selection)));
    UpdateDlg();
}


#if 0
void
SystemConfigDlg::OnWarnIO( wxCommandEvent& WXUNUSED(event) )
{
    bool checked = hWarnIO->IsChecked();
    iomgr_set_warnio(checked ? 1 : 0);
    UpdateDlg();
}
#endif


// used for all dialog button presses
void
SystemConfigDlg::OnButton(wxCommandEvent& event)
{
    switch (event.GetId()) {

        case ID_BTN_REVERT:
            m_config = new SystemConfig(*System3300::GetConfig());  // start with current state
            UpdateDlg();        // select current options
            hBtnRevert->Disable();
            break;

        case wxID_OK:
            SaveDefaults();         // save location & size of dlg
            System3300::SetConfig(*m_config);
            delete m_config; m_config = NULL;
            EndModal(EMU_OK);
            break;

        case wxID_CANCEL:
            SaveDefaults();         // save location & size of dlg
            delete m_config; m_config = NULL;
            EndModal(EMU_FAIL);
            break;

        default:
            wxASSERT(0);
            break;
    }
}


// save dialog options to the config file
void
SystemConfigDlg::SaveDefaults()
{
    const wxString subgroup("ui/configdlg");

    // save position and size
    ConfigWriteWinGeom(this, subgroup);
}


void
SystemConfigDlg::GetDefaults()
{
    // see if we've established a favored location and size
    const wxString subgroup("ui/configdlg");
    ConfigReadWinGeom(this, subgroup);
}
