/////////////////////////////////////////////////////////////////////////////
// UiSystem.cpp
// Modified from wxWindows sample program, drawing.cpp, by Robert Roebling.
//
// This file contains the entry point for the emulator.
/////////////////////////////////////////////////////////////////////////////

// ============================================================================
// declarations
// ============================================================================

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include "UiSystem.h"
#include "UiSystemConfigDlg.h"
#include "UiFrontPanel.h"

#include "system3300.h"

#include "wx/cmdline.h"         // req'd by wxCmdLineParser
#include "wx/tokenzr.h"         // req'd by wxStringTokenizer
#include "wx/settings.h"        // req'd by GetMetric()

#include "wx/filename.h"
#include "wx/utils.h"           // time/date stuff

// ----------------------------------------------------------------------------
// global variables
// ----------------------------------------------------------------------------

wxDateTime g_timezero;  // time program started


// ============================================================================
// implementation
// ============================================================================

// Create a new application object: this macro will allow wxWindows to
// create the application object during program execution (it's better
// than using static object for many reasons) and also declares the
// accessor function wxGetApp() which will return the reference of the
// right type (i.e. MyApp and not wxApp)
IMPLEMENT_APP(MyApp)

BEGIN_EVENT_TABLE(MyApp, wxApp)
    EVT_IDLE (MyApp::OnIdle)
END_EVENT_TABLE()

// ----------------------------------------------------------------------------
// the application class
// ----------------------------------------------------------------------------

wxString      MyApp::m_app_home;        // path to application home directory
wxFileConfig* MyApp::m_config;          // file configuration object

MyApp::term_state_t MyApp::m_term_state = MyApp::RUNNING;
bool MyApp::m_do_reconfig = false;

// `Main program' equivalent: the program execution "starts" here
bool
MyApp::OnInit()
{
#if 0
    // read wxToolBar notes to see why this is done
    if (wxTheApp->GetComCtl32Version() >= 600 && ::wxDisplayDepth() >= 32)
        wxSystemOptions::SetOption(wxT("msw.remap"), 2);
#endif
    InitStatics();      // init static class members

    // path to executable
    wxFileName exe_path(argv[0]);

#ifdef __VISUALC__
    // with ms visual c++, there is a Debug directory and a Release directory.
    // these are one below the anticipated real location where the exe will
    // live, so if we detect we are running from there, we raise the directory
    // one notch.
    int dircount = exe_path.GetDirCount();
    wxArrayString dirnames = exe_path.GetDirs();
    if (dirnames[dircount-1].Lower() == "debug" ||
        dirnames[dircount-1].Lower() == "release") {
        exe_path.AppendDir("..");
        exe_path.Normalize();
    }
#endif

#ifdef __WXMAC__
    // the mac bundle wants references to the Resources directory,
    // not the directory where the application lives
    exe_path.AppendDir("..");
    exe_path.AppendDir("Resources");
    exe_path.Normalize();
#endif

    m_app_home = exe_path.GetPath(wxPATH_GET_VOLUME);
    SetTermState(RUNNING);

    // configuration file stuff
    SetVendorName("Slash & Burn Software");     // not actually used by wxFileConfig
    SetAppName("emu3300");

#ifdef __WXMSW__
    m_config = new wxFileConfig(
                        wxEmptyString,                  // appName
                        wxEmptyString,                  // vendorName
                        m_app_home + "\\emu3300.ini",   // localFilename
                        wxEmptyString,                  // globalFilename
                        wxCONFIG_USE_LOCAL_FILE
                );
#elif defined(__WXMAC__)
    // put emu3300.ini file in same directory as the executable
    // (on the Mac it lives under ~/Library/Preferences)
    wxFileName ini_path("~/Library/Preferences/emu3300.ini");
    ini_path.Normalize();
    m_config = new wxFileConfig("", "", ini_path.GetFullPath());
    wxConfigBase::Set(m_config);
#endif

    GetGlobalDefaults();

    // needed so we can compute a time difference to get ms later
    g_timezero = wxDateTime::UNow();

    // set up emulator state
    System3300 sys;
    sys.Init();

    // create the frame containing the front panel interface
    SetTopWindow(new UiFrontPanel());

    // must call base class version to get command line processing
    // if FALSE, the app terminates
    return wxApp::OnInit();
}


// the user requests a change in configuration from the UiFrontPanel.
// however, doing so often requires a tear down and rebuild of all the
// components.  destroying the frontpanel instance and then returning
// to it is uncouth.  instead, when the user wants to reconfigure, this
// function is called and that desire is noted, and we return immediately.
// later on, in the OnIdle code, this flag is checked and the reconfiguration
// happens then.
void
MyApp::Reconfigure()
{
    m_do_reconfig = true;
}


// this is called whenever the wxWidget event queue is empty,
// indicating there is no work pending.
void
MyApp::OnIdle(wxIdleEvent &event)
{
    System3300 sys;

    if (m_do_reconfig) {
        m_do_reconfig = false;
        SystemConfigDlg dlg(NULL);
        int status = dlg.ShowModal();
    }

    switch (GetTermState()) {
        case RUNNING:
            // this is the normal case during emulation
            sys.EmulateTimeslice();
            event.RequestMore(true);        // give more idle events
            break;
        case TERMINATING:
            // we've been signalled to shut down the universe.
            // change the flag to know we've already cleaned up, in case
            // we receive another OnIdle call.
            SetTermState(TERMINATED);
            sys.Cleanup();
            (UiFrontPanel*)(wxTheApp->GetTopWindow())->Destroy();
            break;
        case TERMINATED:
            // do nothing -- we already requested a cleanup
            break;
        default:
            wxFAIL;
            break;
    }
}


// called to signal we want to end things.  at this point nothing has been
// shut down.  This can be called from within a subordinate window; we
// can't destroy anything now, otherwise we'd return to an object that was
// deleted already.  Instead, we just set a flag indicating we want to shut
// down, and the OnIdle() handler will clean things up from a safe point later.
void
MyApp::Terminate()
{
    SaveGlobalDefaults();

    // flag the OnIdle handler to clean up next time it gets a chance
    SetTermState(TERMINATING);
}


// This is called just before quitting the entire app,
// but before wxWindows cleans up its internal resources.
int
MyApp::OnExit()
{
    delete m_config;
    m_config = NULL;

    return 0;
}


// set the command line parsing options
void
MyApp::OnInitCmdLine(wxCmdLineParser& parser)
{
    // set default wx options
    wxApp::OnInitCmdLine(parser);

    parser.DisableLongOptions();        // -foo, not --foo

    // add options specific to this app
    parser.AddOption("s", "script", "script file to load on startup", wxCMD_LINE_VAL_STRING);
}


// after the command line has been parsed, decode what it finds
bool
MyApp::OnCmdLineParsed(wxCmdLineParser& parser)
{
    // let base class handle its defaults
    bool OK = wxApp::OnCmdLineParsed(parser);

    if (OK) {
        // do my processing here
        // UiAlert("Command line items: %d", parser.GetParamCount());
#ifdef WANG2200
        wxString filename;
        if (parser.Found("s", &filename)) {
            const int io_addr = 0x01;   // default keyboard device
            IOcard_keyboard::invoke_script(io_addr, filename.c_str());
        }
#endif
    }

    return OK;
}


void
MyApp::GetGlobalDefaults()
{
    wxString subgroup("..");
    wxString foo;

    bool b = ConfigReadStr(subgroup, "configversion", &foo);
    if (b && (foo != "1")) {
        UiAlert("Configuration file version '%s' found.\n"
                 "Version '1' expected.\n"
                 "Attempting to read the config file anyway.\n", foo.c_str());
    }

    for(int i=0; i<FILEREQ_NUM; i++) {

        subgroup = m_IniGroup[i];
        long v;

        bool b = ConfigReadStr(subgroup, "directory", &foo);
        m_FileDir[i] = (b) ? foo : ".";

        b = ConfigReadStr(subgroup, "filterindex", &foo);
        m_FileFilterIdx[i] = (b && foo.ToLong(&v)) ? v : 0;

        m_Filename[i] = "";     // intentionally don't save this
    }
}


void
MyApp::SaveGlobalDefaults()
{
    wxString subgroup("..");
    wxString version("1");

    ConfigWriteStr(subgroup, "configversion", &version );

    for(int i=0; i<FILEREQ_NUM; i++) {
        subgroup = m_IniGroup[i];
        ConfigWriteStr(subgroup, "directory", &m_FileDir[i]);
        ConfigWriteInt(subgroup, "filterindex", m_FileFilterIdx[i]);
    }
}


// ----------------------------------------------------------------------------
// uniform file dialog
// ----------------------------------------------------------------------------

wxString MyApp::m_FileDir[FILEREQ_NUM];         // dir where files come from
wxString MyApp::m_Filename[FILEREQ_NUM];        // most recently chosen
wxString MyApp::m_FileFilter[FILEREQ_NUM];      // suffix filter string
int      MyApp::m_FileFilterIdx[FILEREQ_NUM];   // which filter was chosen
wxString MyApp::m_IniGroup[FILEREQ_NUM];        // name in .ini file

void
MyApp::InitStatics()
{
    m_FileDir[FILEREQ_SCRIPT]       = ".";
    m_Filename[FILEREQ_SCRIPT]      = "";
    m_FileFilterIdx[FILEREQ_SCRIPT] = 0;
    m_FileFilter[FILEREQ_SCRIPT]    = "script files (*.w33)"
                                      "|*.w22|text files (*.txt)"
                                      "|*.txt|All files (*.*)|*.*";
    m_IniGroup[FILEREQ_SCRIPT]      = "ui/script";

    m_FileDir[FILEREQ_HEX]          = ".";
    m_Filename[FILEREQ_HEX]         = "";
    m_FileFilterIdx[FILEREQ_HEX]    = 0;
    m_FileFilter[FILEREQ_HEX]       = "HEX (*.hex)|*.hex"
                                       "|Any file (*.*)|*.*";
    m_IniGroup[FILEREQ_HEX]         = "ui/hex";

    m_FileDir[FILEREQ_PTAPE_RDR]        = ".";
    m_Filename[FILEREQ_PTAPE_RDR]       = "";
    m_FileFilterIdx[FILEREQ_PTAPE_RDR]  = 0;
    m_FileFilter[FILEREQ_PTAPE_RDR]     = "BIN (*.bin)|*.bin"
                                         "|Any file (*.*)|*.*";
    m_IniGroup[FILEREQ_PTAPE_RDR]       = "ui/papertape";

    m_FileDir[FILEREQ_PTAPE_PUN]        = ".";
    m_Filename[FILEREQ_PTAPE_PUN]       = "";
    m_FileFilterIdx[FILEREQ_PTAPE_PUN]  = 0;
    m_FileFilter[FILEREQ_PTAPE_PUN]     = "BIN (*.bin)|*.bin"
                                         "|Any file (*.*)|*.*";
    m_IniGroup[FILEREQ_PTAPE_PUN]       = "ui/papertape";

    m_FileDir[FILEREQ_CTAPE]        = ".";
    m_Filename[FILEREQ_CTAPE]       = "";
    m_FileFilterIdx[FILEREQ_CTAPE]  = 0;
    m_FileFilter[FILEREQ_CTAPE]     = "C33 (*.c33)|*.c33"
                                     "|Any file (*.*)|*.*";
    m_IniGroup[FILEREQ_CTAPE]       = "ui/cassettetape";

    m_FileDir[FILEREQ_PRINTER]       = ".";
    m_Filename[FILEREQ_PRINTER]      = "";
    m_FileFilterIdx[FILEREQ_PRINTER] = 0;
    m_FileFilter[FILEREQ_PRINTER]    = "Text Files (*.txt)|*.txt"
                                       "|All files (*.*)|*.*";
    m_IniGroup[FILEREQ_PRINTER]      = "ui/printer";

    // wxFileConfig object
    m_config = NULL;
}


// ask user to provide a file location
int
MyApp::FileReq(int requestor, wxString title, int readonly, wxString *fullpath)
{
    int style;

    wxASSERT(requestor >= 0 && requestor < FILEREQ_NUM);

    style = (readonly) ? (wxFD_OPEN /*| wxHIDE_READONLY*/)
                       : (wxFD_SAVE | wxFD_OVERWRITE_PROMPT);

    // get the name of a file to execute
    wxFileDialog dialog (0,
            title,
            m_FileDir[requestor],       // default directory
            m_Filename[requestor],      // default file
            m_FileFilter[requestor],    // file suffix filter
            style);
    dialog.SetFilterIndex(m_FileFilterIdx[requestor]);

    if (dialog.ShowModal() == wxID_OK) {
        // remember what and where we selected
        m_FileDir[requestor]       = dialog.GetDirectory();
        m_Filename[requestor]      = dialog.GetFilename();
        m_FileFilterIdx[requestor] = dialog.GetFilterIndex();
        *fullpath = dialog.GetPath();
        return FILEREQ_OK;
    }
    return FILEREQ_CANCEL;
}


// ========================================================================
// Help/About functions
// ========================================================================

wxMenu*
MyApp::MakeHelpMenu()
{
    wxMenu *menuHelp = new wxMenu;
//  menuHelp->Append(Help_Quickstart,  "&Quickstart",      "Information for new users of WangEmu");
//  menuHelp->Append(Help_Configure,   "&Configuration",   "Information about configuring the emulator");
//  menuHelp->Append(Help_Keyboard,    "&Keyboard",        "Information about how the 2200 keyboard is mapped onto yours");
//  menuHelp->Append(Help_Menus,       "&Menus",           "Information about the emulator menu system");
//  menuHelp->Append(Help_Printer,     "&Printer",         "Information about the using the emulated printer");
//  menuHelp->Append(Help_Script,      "&Script",          "Information about loading files into the emulator");
    menuHelp->Append(Help_Website,     "&Website",         "Open a browser to the emulator's web site");
    #ifndef __WXMAC__
    // this isn't needed on the Mac because the About menu item gets forcibly moved to another place
    menuHelp->AppendSeparator();
    #endif
    menuHelp->Append(Help_About,       "&About...",        "Information about the program");

    return menuHelp;
}


void
MyApp::OnHelp_Launcher(wxCommandEvent& event)
{
    wxString helpfile;
    bool absolute = FALSE;
    switch (event.GetId()) {
//      case Help_Quickstart:   helpfile = "quickstart.html";      break;
//      case Help_Configure:    helpfile = "configure.html";       break;
//      case Help_Keyboard:     helpfile = "keyboard.html";        break;
//      case Help_Menus:        helpfile = "menus.html";           break;
//      case Help_Printer:      helpfile = "printer.html";         break;
//      case Help_Script:       helpfile = "script.html";          break;
        case Help_Website:      helpfile = "http://www.thebattles.net/wang/3300.html"; absolute = true; break;
        default:
            wxFAIL;
            return;
    }

    wxChar sep(wxFileName::GetPathSeparator());
    wxString target_file = (absolute) ? helpfile
                                      : ( "file://" + m_app_home + sep + "html" + sep + helpfile );
    ::wxLaunchDefaultBrowser(target_file);
}


void
MyApp::OnHelp_About(wxCommandEvent& WXUNUSED(event))
{
    wxString msg;
    msg.Printf( "Wang 3300 Emulator\n"
                "Version 0.3, December 29, 2007\n"
                "Copyright (c) Jim Battle, Slash & Burn Software, 2007\n\n"
                "Built with %s\n\n"
                "Thanks for crucial information and encouragement:\n"
                "    Steve Witham\n"
                "    Tom Lake"
                , wxVERSION_STRING
              );

    // unfortunately, centering the text doesn't work on windows
    wxMessageDialog dialog(NULL, msg, "About this program",
                            wxOK | wxICON_INFORMATION | wxCENTRE);
#ifndef __WXMSW__
    dialog.CenterOnParent();    // this doesn't seem to work (on Win32 at least)
#endif
    dialog.ShowModal();
}

// ========================================================================
//    User Interface utility functions
// ========================================================================

// fetch an association from the configuration file
bool
ConfigReadStr(const wxString &subgroup, const wxString &key, wxString *val,
              const wxString *defaultval)
{
    MyApp::m_config->SetPath( "/emu3300/config-0/" + subgroup);
    bool b = MyApp::m_config->Read(key, val);
    if (!b && (defaultval != NULL) && (val != defaultval))
        *val = *defaultval;
    return b;
}

bool
ConfigReadInt(const wxString &subgroup, const wxString &key, int *val,
               const int defaultval)
{
    wxString valstr;
    long v = 0;
    bool b = ConfigReadStr(subgroup, key, &valstr);
    if (b)
        b = valstr.ToLong(&v, 0);  // 0 means allow hex and octal notation too
    *val = (b) ? (int)v : defaultval;
    return b;
}

void
ConfigReadBool(const wxString &subgroup, const wxString &key, bool *val,
               const bool defaultval)
{
    wxString valstr;
    int v;
    bool b = ConfigReadInt(subgroup, key, &v, ((defaultval) ? 1:0) );
    if (b && (v >= 0) && (v <= 1))
        *val = (v==1);
    else
        *val = defaultval;
}

// read the geometry for a window and if it is reasonable, use it, otherwise
// use the supplied default (if any).  the reason for the client_size control
// is that all the windows used client size, but for some reason, the
// printpreview window would shrink vertically on each save/restore cycle
// by 20 pixels.  using GetSize/SetSize fixed that problem for printpreview.
// It would have been OK for every window to use that instead I suppose, but
// for people who already had a .ini file, it would have messed up their
// settings a bit on the changeover.
void
ConfigReadWinGeom(wxWindow *wxwin, const wxString &subgroup,
                  wxRect *default_geom, bool client_size)
{
    long x=0, y=0, w=0, h=0;    // just for lint
    wxString valstr;

    bool b = ConfigReadStr(subgroup, "window", &valstr);
    if (b) {
        wxStringTokenizer stkn(valstr, ",");
        b = (stkn.CountTokens() == 4);  // there should be four items
        if (b) {
            wxString str_x = stkn.GetNextToken();
            wxString str_y = stkn.GetNextToken();
            wxString str_w = stkn.GetNextToken();
            wxString str_h = stkn.GetNextToken();
            b = str_x.IsNumber() && str_y.IsNumber() &&
                str_w.IsNumber() && str_h.IsNumber();
            if (b) {
                str_x.ToLong( &x, 0 );
                str_y.ToLong( &y, 0 );
                str_w.ToLong( &w, 0 );
                str_h.ToLong( &h, 0 );
            }
        }
    }

    if (!b) {
        // the specified geometry was bad; use the supplied default
        if (default_geom == NULL)
            return;     // nothing we can do
        x = default_geom->GetX();
        y = default_geom->GetY();
        w = default_geom->GetWidth();
        h = default_geom->GetHeight();
    }

    // sanity check window position
    int screen_w = wxSystemSettings::GetMetric(wxSYS_SCREEN_X);
    int screen_h = wxSystemSettings::GetMetric(wxSYS_SCREEN_Y);

    // if origin is off screen, move it on screen
    if (x < 0) { x = 0; }
    if (y < 0) { y = 0; }
    if ((x > screen_w-4) || (y > screen_h-4))   // at least a tiny nub must show
        return;

    // don't let the window be bigger than the screen
    if (w > screen_w)
        w = screen_w;
    if (h > screen_h)
        h = screen_h;

    // now move and resize the window
    wxPoint pt(x,y);
    wxSize  sz(w,h);
    wxwin->Move(pt);
    if (client_size)
        wxwin->SetClientSize(sz);
    else
        wxwin->SetSize(sz);
}

// this function classifies the supplied filename as being either
// relative (returns 0) or absolute (returns 1).
int
HOST_IsAbsolutePath(char *name)
{
    wxASSERT(name != NULL);
    wxFileName filename(name);
    return filename.IsAbsolute();
}


// send a string association to the configuration file
void
ConfigWriteStr(const wxString &subgroup, const wxString &key, const wxString *val)
{
    MyApp::m_config->SetPath( "/emu3300/config-0/" + subgroup);
    bool b = MyApp::m_config->Write(key, *val);
    wxASSERT(b);
    b = b;      // keep lint happy
}

// send an integer association to the configuration file
void
ConfigWriteInt(const wxString &subgroup, const wxString &key, const int val)
{
    wxString foo;
    foo.Printf("%d", val);
    ConfigWriteStr(subgroup, key, &foo);
}

// send a boolean association to the configuration file
void
ConfigWriteBool(const wxString &subgroup, const wxString &key, const bool val)
{
    int foo = (val) ? 1 : 0;
    ConfigWriteInt(subgroup, key, foo);
}

// write out the geometry for a window
void
ConfigWriteWinGeom(wxWindow *wxwin, const wxString &subgroup, bool client_size)
{
    wxASSERT(wxwin != NULL);

    int x, y, w, h;
    wxwin->GetPosition(&x, &y);
    if (client_size)
        wxwin->GetClientSize(&w, &h);
    else
        wxwin->GetSize(&w, &h);

    wxString prop;
    prop.Printf("%d,%d,%d,%d", x,y,w,h);
    ConfigWriteStr(subgroup, "window", &prop);
}

// fetch an association from the configuration file
// used by the core emulator
int
UiConfigReadStr(const char *subgroup, const char *key, char *val)
{
    wxString foo;
    // make sure malformed config files don't bomb caller
    bool b = ConfigReadStr(subgroup, key, &foo);
    if (b) {
        strncpy(val, foo, 255);
        return 1;
    }
    return 0;   // failed
}


// send an association to the configuration file
// used by the core emulator
void
UiConfigWriteStr(const char *subgroup, const char *key, const char *val)
{
    wxString foo(val);
    ConfigWriteStr(subgroup, key, &foo);
}


// if something goes wrong and the only thing to do is report an
// error, this is the way to do it.
void
UiAlert(const char *fmt, ...)
{
    char buff[1000];
    wxString info;
    va_list args;

    va_start(args, fmt);
#ifdef __WXMAC__
    vsprintf(buff, fmt, args);  //FIXME to prevent buffer overruns
#else
    _vsnprintf(buff, sizeof(buff), fmt, args);
#endif
    va_end(args);

    info = buff;
    wxMessageDialog dialog(NULL, info, "Warning");
    dialog.ShowModal();
}


// get a YES/NO confirmation
int
UiConfirm(const char *fmt, ...)
{
    char buff[1000];
    wxString info;
    va_list args;
    long style = wxYES | wxNO | wxNO_DEFAULT | wxICON_EXCLAMATION;

    va_start(args, fmt);
#ifdef __WXMAC__
    vsprintf(buff, fmt, args);  //FIXME to prevent buffer overruns
#else
    _vsnprintf(buff, sizeof(buff), fmt, args);
#endif
    va_end(args);

    info = buff;
    wxMessageDialog dialog(NULL, info, "Warning", style);
    int rv = dialog.ShowModal();
    return (rv == wxID_YES);
}


// return the time in milliseconds as a 64b signed integer
int64
UiGetTimeMs(void)
{
    // FIXME: this is horrible.  is there a better way?
    wxDateTime now = wxDateTime::UNow();
    wxTimeSpan diff = now - g_timezero;
    wxLongLong delta = diff.GetMilliseconds();
    uint32 low  = delta.GetLo();
    uint32 high = delta.GetHi();
    int64 time_ms = ((int64)high << 32) | low;
    return time_ms;
}


// called periodically by the core emulation so UI events can occur
void
UiYield(void)
{
    wxTheApp->Yield(true);      // simply returns if things get recursive
}


// go to sleep for approximately ms milliseconds before returning
void
UiSleep(unsigned int ms)
{
    wxMilliSleep(ms);
}

// set simulation time for informative display
void
UiSetSimSeconds(unsigned long secs, float relative_speed)
{
    wxString str;
    wxString format = (relative_speed >= 10.0f) ?
                      "Sim time: %d seconds, %3.0fx"
                    : "Sim time: %d seconds, %3.1fx";
    str.Printf( format, secs, relative_speed );

    UiFrontPanel *tf = (UiFrontPanel*)(wxTheApp->GetTopWindow());
    tf->m_statbar->SetStatusText(str);
}
