/////////////////////////////////////////////////////////////////////////////
// UiFrontPanel.cpp
// Modified from wxWindows sample program, drawing.cpp, by Robert Roebling.
//
// This file contains the GUI for the CPU front panel.
/////////////////////////////////////////////////////////////////////////////

// TODO:
//  * the front panel diagram shows the S register organized differently
//    than the register layout.  specifically, the carry bit it split off
//    from the others ("Ca").  perhaps it wasn't settable by the ENTER mode?
//  * the front panel has indicator lamps "Stop" and "I/O" as well.
//  * the 0-9A-F shortcut could perhaps be generalized to controlling
//    other buttons, eg,
//         0-9a-f: hex chars for data buttons
//         ABCSMZ?: register (? for core, but perhaps there is something better)
//                  caps distinguish ABC from the hex data abc,
//                  or perhaps we just qualify by where the mouse is hovered.
//         RDE: run, display, enter mode (again, DE vs de)
//         LSQG: load, step execute, go commands
//  * there is a shortcut where you can type [0-9A-F] while the mouse is
//    hovered over the data buttons to quickly enter hex values.
//    unfortunately, there is a bell after each received keystroke.
//    this is because we intercept the KEY_DOWN event and swallow it,
//    but there is a subsequent EVT_CHAR (and probably KEY_UP) that
//    can't be intercepted the same way, so they end up percolating
//    through the event handler hierarchy and then causing a default
//    auction: ringing the bell.
//  * there are a number of photographs of actual 3300 units, and some
//    illustrations as well.  there are two flavors of front panel.
//    one has the following layout for the buttons:
//        (command) (mode) (data) (register)
//    the other has this layout:
//        (register) (data) (mode) (commnad)
//    there are also three different logo styles, although one appears only
//    in one place (a rough diagram in the the developers' manual), so we'll
//    discount that one.  These two styles of logo correlate with the two
//    styles of button layout.  perhaps one was used on prototype or maybe
//    early prodution, and the other style came later.

// ----------------------------------------------------------------------------
// headers
// ----------------------------------------------------------------------------

#include "UiSystem.h"
#include "UiFrontPanel.h"

#include "system3300.h"
#include "cpu3300.h"
#include "scheduler.h"
#include "term3315.h"
#include "intel_hex.h"          // for reading/writing intel .hex files
#include "bootstrap.h"          // boot loader images

#include "wx/tglbtn.h"          // wxToggleButton

#define LAMP_REFRESH_PERIOD_MS 33.333   // 30 Hz

// ========================================================================
// Front Panel interface
// ========================================================================

enum
{
    File_Load_Hex = 1,
    File_Save_Hex,
    File_Quit,

    Cpu_Reset,

    System_Reconfigure,

    Timer_Refresh,      // ID for timer callbacks
};


BEGIN_EVENT_TABLE(UiFrontPanel, wxFrame)
    EVT_MENU         (File_Load_Hex,       UiFrontPanel::OnLoadHex)
    EVT_MENU         (File_Save_Hex,       UiFrontPanel::OnSaveHex)
    EVT_MENU         (File_Quit,           UiFrontPanel::OnQuit)
    EVT_MENU         (Cpu_Reset,           UiFrontPanel::OnCpuReset)
    EVT_MENU         (System_Reconfigure,  UiFrontPanel::OnSystemReconfigure)

    // non-menu event handlers
    EVT_CLOSE        (UiFrontPanel::OnClose)
    EVT_BUTTON       (wxID_ANY,      UiFrontPanel::OnButton)
    EVT_TOGGLEBUTTON (wxID_ANY,      UiFrontPanel::OnToggleButton)
    EVT_TIMER        (Timer_Refresh, UiFrontPanel::OnTimer)

    // help menu items do whatever they need to do
    HELP_MENU_EVENT_MAPPINGS()

END_EVENT_TABLE()


enum {
    ctlId_Clear,        // the button to clear machine state

    ctlId_dispZ0 = 0x10,    // Z0 to Z7
    ctlId_dispA0 = 0x18,    // A0 to A7
    ctlId_dispS0 = 0x20,    // S0 to S7
    ctlId_dispB0 = 0x30,    // B0 to B7
    ctlId_dispC0 = 0x38,    // C0 to C7
    ctlId_dispM0 = 0x40,    // M0 to M7

    ctlId_regB    = 0x50,    // B register select
    ctlId_regC    = 0x51,    // C register select
    ctlId_regZ    = 0x52,    // Z register select
    ctlId_regA    = 0x53,    // A register select
    ctlId_regS    = 0x54,    // S register select
    ctlId_regM    = 0x55,    // S register select
    ctlId_regCore = 0x56,    // core memory select

    ctlId_data    = 0x60,    // data value buttons

    ctlId_mode_run     = 0x68,      // "Run" mode button
    ctlId_mode_display = 0x69,      // "Display" mode button
    ctlId_mode_enter   = 0x6a,      // "Enter" mode button

    ctlId_command_load = 0x6b,      // "Load" command button
    ctlId_command_step = 0x6c,      // "Step" command button
    ctlId_command_exq  = 0x6d,      // "ExQ" (execute) command button
    ctlId_command_go   = 0x6e,      // "Go" command button
};

static int ids[6] =
    { ctlId_dispZ0, ctlId_dispA0, ctlId_dispS0, ctlId_dispB0, ctlId_dispC0, ctlId_dispM0 };
static char *groups[6] =
    { "Z", "A", "Status", "B", "C", "Memory" };
static char *bin_labels[8] =
    { "1", "2", "4", "8", "10", "20", "40", "80" };
static char *s_labels[7] =
    { "0", "1", "V", "D", "Z", "C", "P" };

// constructor
UiFrontPanel::UiFrontPanel() :
    wxFrame((wxFrame*)NULL, wxID_ANY, "Wang 3300",
            wxDefaultPosition, wxDefaultSize,
            wxDEFAULT_FRAME_STYLE | wxNO_FULL_REPAINT_ON_RESIZE)
{
    // read in configuration information
    GetDefaults();

    // build menus
    MakeMenubar();

    // build startbar
    m_statbar = new wxStatusBar(this);
    m_statbar->SetFieldsCount(1);
    SetStatusBar(m_statbar);

    // create all the GUI do-dads that constitute the front panel
    MakeFrontPanel();

    // refresh the lamps at 60 Hz of real time
    m_refresh_timer = new wxTimer(this, Timer_Refresh);
    m_refresh_timer->Start(LAMP_REFRESH_PERIOD_MS, wxTIMER_CONTINUOUS);

    // refresh the lamps at 60 Hz of simulated time
    System3300 sys;
    const int refresh_ticks = TIMER_MS(LAMP_REFRESH_PERIOD_MS);
    sys.scheduler()->TimerCreate(refresh_ticks, *this, &UiFrontPanel::RefreshCb, 0);
}


// destructor
UiFrontPanel::~UiFrontPanel()
{
    // free the bitmaps that hold the lamp images
    for(int p=0; p<101; p++) {
        delete m_lamps[p];
        m_lamps[p] = NULL;
    }

    delete m_refresh_timer;
    m_refresh_timer = NULL;
}


// read an intel .hex file into memory
void
UiFrontPanel::OnLoadHex(wxCommandEvent& WXUNUSED(event))
{
    // open a dialog to get a filename
    wxString name;
    if (MyApp::FileReq(FILEREQ_HEX, "Intel Hex File to Read", 1, &name) != FILEREQ_OK)
        return;
    wxASSERT(!IsEmpty(name));

    HexFile hf(name.c_str(), 'r');
    if (!hf.Opened()) {
        UiAlert("Error opening file '%s'", name.c_str());
        return;
    }

    if (!hf.IsLegal()) {
        UiAlert("Bad file format, line %d", hf.GetBadLine());
        return;
    }

    System3300 sys;
    while (!hf.IsDone()) {
        int addr;
        int byte = hf.GetNextByte(addr);
        if (byte == -1) // EOF
            return;
        if (byte == -2) { // bad file format
            wxFAIL;     // should have caught this earlier
        }
        sys.cpu()->SetMemory(addr, byte);
    }
}


// save memory image as an intel .hex file
void
UiFrontPanel::OnSaveHex(wxCommandEvent& WXUNUSED(event))
{
    // open a dialog to get a filename
    wxString name;
    if (MyApp::FileReq(FILEREQ_HEX, "Intel Hex File to Save", 0, &name) != FILEREQ_OK)
        return;
    wxASSERT(!IsEmpty(name));

    HexFile hf(name.c_str(), 'w');
    if (!hf.Opened()) {
        UiAlert("Error opening file '%s'", name.c_str());
        return;
    }

    System3300 sys;
    int ram_bytes = 1024 * (sys.GetConfig()->GetRamKB());
    for(int a=0x0000; a < ram_bytes; a++) {
        int byte = sys.cpu()->GetMem8(a, false);
        hf.SaveByte(byte, a);
    }
}


// this even occurs when "Quit" is selected from the menu.
// since we want it to be idential to clicking the system
// "close this window" box, we just call that other function.
void
UiFrontPanel::OnQuit(wxCommandEvent& WXUNUSED(event))
{
    Close(true);
}


// this occurs when the system is requesting that the window
// be shut down.
void
UiFrontPanel::OnClose(wxCloseEvent& WXUNUSED(event))
{
    // there is a lot to clean up
    MyApp::Terminate();
}


// save off window geometry just before shuting down
bool
UiFrontPanel::Destroy()
{
    SaveDefaults();             // save config options
    return wxFrame::Destroy();  // let superclass handle real work
}


// respond to button interaction
void
UiFrontPanel::OnButton(wxCommandEvent &event)
{
    int id = event.GetId();
    int reg_enum = -1;
    int term_count = System3300::NumTerms();

    switch (id) {

        case ctlId_Clear:
            {
                System3300 sys;
                sys.cpu()->Clear();
                for(int i=0; i<term_count; i++) {
                    sys.term(i)->Clear();
                }
            }
            return;

        case ctlId_command_load:
            if (GetModeButtons() != ctlId_mode_enter) {
                UiAlert("LOAD is ignored unless in ENTER mode");
            } else {
                // manual describes the function like this:
                //    Loads first block of program from peripheral device
                //    (Enter Mode) (If ahrdware bootstrap optoin is installed)
                // FIXME: that isn't exactly what we are doing here.
                //        nevertheless, it is still useful.
                System3300 sys;
#if 1
                sys.ReadByteProgram(bootstrap);
                sys.cpu()->SetReg(cpu3300::FPREG_B, 0x00);
                sys.cpu()->SetReg(cpu3300::FPREG_C, 0x00);
#elif 0
                sys.ReadByteProgram(tcboot);
                sys.cpu()->SetReg(cpu3300::FPREG_B, 0x2F);
                sys.cpu()->SetReg(cpu3300::FPREG_C, 0x40);
#else
                wxString name("BASIC_28K.hex");
                HexFile hf(name.c_str(), 'r');
                if (!hf.Opened()) { UiAlert("Error opening file '%s'", name.c_str()); return; }
                if (!hf.IsLegal()) { UiAlert("Bad file format, line %d", hf.GetBadLine()); return; }
                while (!hf.IsDone()) {
                    int addr;
                    int byte = hf.GetNextByte(addr);
                    if (byte == -1) break; // EOF
                    if (byte == -2) { wxFAIL; } // should have been caught earlier
                    sys.cpu()->SetMemory(addr, byte);
                }
                sys.cpu()->SetReg(cpu3300::FPREG_B, 0x42);
                sys.cpu()->SetReg(cpu3300::FPREG_C, 0x00);
#endif
            }
            return;

        case ctlId_command_step:
            if (GetModeButtons() != ctlId_mode_run) {
                UiAlert("STEP is ignored unless in RUN mode");
                return;
            } else {
                // run a single instruction
                System3300 sys;
                sys.cpu()->SetCpuStatus(cpu3300::CPUS_STEP);
            }
            return;

        case ctlId_command_exq:
            if (GetModeButtons() != ctlId_mode_enter) {
                UiAlert("EXQ is ignored unless in ENTER mode");
            } else {
                System3300 sys;
                sys.cpu()->SetReg(cpu3300::FPREG_I, GetDataButtons());
                sys.cpu()->SetCpuStatus(cpu3300::CPUS_EXQ);
            }
            return;

        case ctlId_command_go:
            if (GetModeButtons() != ctlId_mode_run) {
                UiAlert("GO is ignored unless in RUN mode");
                return;
            } else {
                System3300 sys;
                sys.cpu()->SetCpuStatus(cpu3300::CPUS_RUNNING);
                // the next time slice the CPU will start free-running
            }
            return;

        case ctlId_regB:
            reg_enum = cpu3300::FPREG_B;
            break;
        case ctlId_regC:
            reg_enum = cpu3300::FPREG_C;
            break;
        case ctlId_regZ:
            reg_enum = cpu3300::FPREG_Z;
            break;
        case ctlId_regA:
            reg_enum = cpu3300::FPREG_A;
            break;
        case ctlId_regS:
            reg_enum = cpu3300::FPREG_S;
            break;
        case ctlId_regM:
            reg_enum = cpu3300::FPREG_M;
            break;
        case ctlId_regCore:
            break;

        default:
            wxFAIL;
            break;
    }

    // one of the register buttons was pressed
    int mode = GetModeButtons();
    if (mode == ctlId_mode_run) {
        UiAlert("Pressing a register button in RUN mode has no effect");
        return; // just ignore it
    } else if (mode == ctlId_mode_enter) {
        System3300 sys;
        if (reg_enum < 0) {
            // set a core memory value
            int addr = (sys.cpu()->GetReg(cpu3300::FPREG_B) << 8) |
                       (sys.cpu()->GetReg(cpu3300::FPREG_C) << 0) ;
            sys.cpu()->SetMemory(addr, GetDataButtons());
            // writing core has the side effect of incrementing BC
            addr++;
            sys.cpu()->SetReg(cpu3300::FPREG_B, (addr >> 8) & 0xFF);
            sys.cpu()->SetReg(cpu3300::FPREG_C, (addr >> 0) & 0xFF);
        } else {
            // set a register value
            sys.cpu()->SetReg(reg_enum, GetDataButtons());
        }
    } else /* if (mode == ctlId_mode_display) */ {
        if (reg_enum < 0) {
            // display a core memory value
            System3300 sys;
            int addr = (sys.cpu()->GetReg(cpu3300::FPREG_B) << 8) |
                       (sys.cpu()->GetReg(cpu3300::FPREG_C) << 0) ;
            int byte = sys.cpu()->GetMem8(addr);
            sys.cpu()->SetReg(cpu3300::FPREG_M, byte);
            // reading core has the side effect of incrementing BC
            addr++;
            sys.cpu()->SetReg(cpu3300::FPREG_B, (addr >> 8) & 0xFF);
            sys.cpu()->SetReg(cpu3300::FPREG_C, (addr >> 0) & 0xFF);
        } else {
            UiAlert("Pressing a register button in DISPLAY mode has no effect");
            return; // just ignore it
        }
    }
}


// return the value 0-255 described by the data toggle buttons
int
UiFrontPanel::GetDataButtons()
{
    int val = 0;

    for(int b=0; b<8; b++) {
        wxWindow *w = FindWindowById(ctlId_data+b, m_panel);
        wxASSERT(w != NULL);
        wxToggleButton *btn = static_cast<wxToggleButton*>(w);
        int depressed = btn->GetValue();
        val += (depressed) ? (1 << b) : 0;
    }

    return val;
}

void
UiFrontPanel::SetDataButtons(int v)
{
    wxASSERT(v >= 0 && v <= 255);
    for(int b=0; b<8; b++) {
        wxWindow *w = FindWindowById(ctlId_data+b, m_panel);
        wxASSERT(w != NULL);
        wxToggleButton *btn = static_cast<wxToggleButton*>(w);
        btn->SetValue((v >> b) & 1);
    }
}


// respond to toggle button interaction
// we don't need to process clicks to the data value bits; the state
// of these buttons is sampled under various other conditions later.
void
UiFrontPanel::OnToggleButton(wxCommandEvent &event)
{
    int id = event.GetId();

    if ((id >= ctlId_data) && (id < ctlId_data+8)) {
        event.Skip();
        return;
    }

    switch (id) {
        case ctlId_mode_run:
        case ctlId_mode_display:
        case ctlId_mode_enter:
            break;
        default:
            wxFAIL;
            break;
    }

    // don't allow clicking an already-selected control
    wxWindow *w = FindWindowById(id, m_panel);
    wxASSERT(w != NULL);
    wxToggleButton *btn = static_cast<wxToggleButton*>(w);
    bool btn_state = btn->GetValue();
    // counterintuitively, we check for false because the state
    // we retrieve is the value after the state inversion.
    if (!btn_state) {
        btn->SetValue(true);
        return;
    }

    // force the other buttons inactive
    SetModeButtons(id);
}


// make the mode buttons exclusive
void
UiFrontPanel::SetModeButtons(int id)
{
    static const int id_list[] = { ctlId_mode_run, ctlId_mode_display, ctlId_mode_enter };
    for(int i=0; i<3; i++) {
        wxWindow *w = FindWindowById(id_list[i], m_panel);
        wxASSERT(w != NULL);
        wxToggleButton *btn = static_cast<wxToggleButton*>(w);
        bool btn_state = btn->GetValue();
        bool match = (id == id_list[i]);
        btn->SetValue(match);
    }
}


// return which of the modes is currently active (returns ctlId value)
int
UiFrontPanel::GetModeButtons()
{
    static const int id_list[] = { ctlId_mode_run, ctlId_mode_display, ctlId_mode_enter };
    for(int i=0; i<3; i++) {
        wxWindow *w = FindWindowById(id_list[i], m_panel);
        wxASSERT(w != NULL);
        wxToggleButton *btn = static_cast<wxToggleButton*>(w);
        if (btn->GetValue())
            return id_list[i];
    }
    wxFAIL;
    return id_list[0];  // keep lint happy
}


// if we are running the CPU unregulated, refresh the lamps according
// to passage of real time
void
UiFrontPanel::OnTimer(wxTimerEvent &event)
{
    System3300 sys;
    if ((event.GetId() == Timer_Refresh) && !sys.CpuSpeedRegulated())
        UpdateLamps();
}


// if we are running the CPU regulated, refresh the lamps according
// to passage of simulated time
void
UiFrontPanel::RefreshCb(int n)
{
    System3300 sys;

    if (sys.CpuSpeedRegulated())
        UpdateLamps();

    // ask for it to happen again
    const int refresh_ticks = TIMER_MS(LAMP_REFRESH_PERIOD_MS);
    sys.scheduler()->TimerCreate(refresh_ticks, *this, &UiFrontPanel::RefreshCb, 0);
}


void
UiFrontPanel::UpdateLamps()
{
#ifdef DIRECT_BLIT
    wxClientDC dst(m_panel);
    wxMemoryDC src;
#endif

    // poll each register for its time averaged intensity
    System3300 sys;
    for(int n=0; n<6; n++) {
        static int reg_enum[6] = { 
                cpu3300::FPREG_Z, cpu3300::FPREG_A, cpu3300::FPREG_S,
                cpu3300::FPREG_B, cpu3300::FPREG_C, cpu3300::FPREG_M
                };
        int *histo = sys.cpu()->GetLedHisto(reg_enum[n]);
        for(int b=7; b>=0; b--) {
            bool status = (n == 2);
            if (status && b == 7)
                continue;       // status has no bit 7
            int ctl_offset = n*8+b;

            // compute the intensity of the lamp.  the thermal lag of the
            // lamp is modeled with a 1st order lowpass filter.
            // this magic constant should depend on LAMP_REFRESH_PERIOD_MS
            const float LAMP_K = 0.80f;
            int impulse = float(histo[b]);
            float new_drive = (  LAMP_K) * float(impulse)
                            + (1-LAMP_K) * m_lampdrive[ctl_offset];
            wxASSERT(new_drive >= 0.0 && new_drive <= 100.0);
            m_lampdrive[ctl_offset] = new_drive;
            int percent = int(new_drive + 0.5f);

            wxStaticBitmap *bm = m_lampctl[ctl_offset];
#ifdef DIRECT_BLIT
            // draw to the client surface directly.
            // right now this isn't working because the control takes priority
            // over what we are drawing.  if the control is set Show(false);
            // it collapses the layout.  thus to make this really work, we'd
            // have to compute the position of all the text labels, static
            // boxes, etc.  we may want to do this eventually to get more
            // control over the layout to be more authentic, but it isn't
            // worth it right now.
            //
            // another thing worth trying is to have a single bitmap containing
            // all the lamp states, and then draw from the right subimage.
            // this saves having to src.SelectObject() 48 times per update.
            //
            // even without this optimization, that is just doing the direct
            // draw, resulting in a 23x speed vs a 19x speed, in unregulated
            // mode.
            src.SelectObject(*m_lamps[percent]);
            wxRect dst_rc(bm->GetRect());
            dst.Blit( dst_rc.GetX(),     dst_rc.GetY(),
                      dst_rc.GetWidth(), dst_rc.GetHeight(),
                      &src, 0, 0);
                
#else
            // update the control and let the control do the draw
            bm->SetBitmap(*m_lamps[percent]);
#endif
        }
    }

    sys.cpu()->InitLedHisto();  // reset the averages
}


void
UiFrontPanel::OnCpuReset(wxCommandEvent& WXUNUSED(event))
{
    System3300 sys;
    sys.cpu()->Reset();

    int term_count = sys.NumTerms();
    for(int i=0; i<term_count; i++) {
        sys.term(i)->Reset();
    }
}


void
UiFrontPanel::OnSystemReconfigure(wxCommandEvent& WXUNUSED(event))
{
    MyApp::Reconfigure();
}


// ---- utility functions ----

// create menubar
void
UiFrontPanel::MakeMenubar()
{
    wxMenu *menuFile = new wxMenu;
    menuFile->Append(File_Load_Hex, "Load &Hex...", "Read in Intel .hex file");
    menuFile->Append(File_Save_Hex, "Save &Hex...", "Save memory image as Intel .hex file");
    menuFile->Append(File_Quit,     "E&xit",        "Exit the program");

    wxMenu *menuCpu = new wxMenu;
    menuCpu->Append(Cpu_Reset, "&Reset", "Reset the CPU");

    wxMenu *menuConfig = new wxMenu;
    menuConfig->Append(System_Reconfigure, "&Configure ...", "Specify system configuration");

    wxMenu *menuHelp = MyApp::MakeHelpMenu();

    wxMenuBar *menuBar = new wxMenuBar;
    menuBar->Append(menuFile,   "&File");
    menuBar->Append(menuCpu,    "&CPU");
    menuBar->Append(menuConfig, "&System");
    menuBar->Append(menuHelp,   "&Help");

    SetMenuBar(menuBar);
}


// create bitmaps of indicator lamps
void
UiFrontPanel::MakeLampImages()
{
    const int lamp_d = 28;  // lamp dimension
    const int lamp_r = 12;  // radius of lit part
    const int lamp_e =  1;  // thickness of lamp edge
#if 1
    wxColor bg = wxColor( 0x40, 0x40, 0x30 );   // background fill
#else
    wxColor bg = wxSystemSettings::GetColour(wxSYS_COLOUR_BTNFACE);
#endif
    wxColor lamp_off  = wxColor( 0x50, 0x50, 0x30 );   // lamp when off
    wxColor lamp_on   = wxColor( 0xFF, 0xFF, 0xB0 );   // lamp when fully on
    wxColor lamp_edge = wxColor( 0x48, 0x48, 0x30 );   // edge of lamp
    for(int p=0; p<101; p++) {
        wxMemoryDC dc;
        wxBitmap lamp(3*lamp_d,3*lamp_d);
        dc.SelectObject(lamp);
        // fill with background color
        dc.SetBrush( bg );
        dc.SetPen( wxPen(bg, 1, wxTRANSPARENT) );   // no pen
        dc.DrawRectangle( 0, 0, 3*lamp_d, 3*lamp_d );   // x,y,w,h
        // draw lamp that is p percent lit
        wxColor fill = wxColor(
            (int)((p/100.0)*(lamp_on.Red()   - lamp_off.Red()  )) + lamp_off.Red(),
            (int)((p/100.0)*(lamp_on.Green() - lamp_off.Green())) + lamp_off.Green(),
            (int)((p/100.0)*(lamp_on.Blue()  - lamp_off.Blue() )) + lamp_off.Blue()
                        );
        dc.SetBrush( fill );
        dc.SetPen( wxPen(lamp_edge, 3*lamp_e, wxSOLID) );
        dc.DrawCircle( 3*lamp_d>>1, 3*lamp_d>>1, 3*lamp_r );  // x,y,r

        // scale it down, such that we get smoother edges
        wxImage img = lamp.ConvertToImage();
        m_lamps[p] = new wxBitmap(img.Rescale(lamp_d, lamp_d, wxIMAGE_QUALITY_HIGH));
    }
}


wxBoxSizer *
UiFrontPanel::MakePanelButton(char *label, int id, wxSize size, bool toggle)
{
    wxStaticText *txt = new wxStaticText(m_panel, wxID_STATIC, label);
    wxButton *btn = (toggle) ?
        (wxButton*)(new wxToggleButton(m_panel, id, "", wxDefaultPosition, size)) :
                    new wxButton(m_panel, id, "", wxDefaultPosition, size) ;
    wxBoxSizer *bs = new wxBoxSizer( wxVERTICAL );
    bs->Add(txt, 0, wxALIGN_CENTER | wxBOTTOM, 4);
    bs->Add(btn, 0, wxALIGN_CENTER | wxBOTTOM, 4);
    return bs;
}


wxBoxSizer *
UiFrontPanel::MakePanelLamp(char *label, int id, int ctl_offset)
{
    wxStaticText *txt = new wxStaticText(m_panel, wxID_STATIC, label);
    wxStaticBitmap *bm = new wxStaticBitmap(m_panel, id, *m_lamps[0]);
    wxBoxSizer *bs = new wxBoxSizer( wxVERTICAL );
    bs->Add(txt, 0, wxALIGN_CENTER | wxBOTTOM, 4);
    bs->Add(bm,  0, wxALIGN_CENTER | wxBOTTOM, 4);
    m_lampctl[ctl_offset] = bm;
    m_lampdrive[ctl_offset] = 0.0f;
    return bs;
}


// create all the GUI do-dads that constitute the front panel
void
UiFrontPanel::MakeFrontPanel()
{
    // create bitmaps of indicator lamps
    MakeLampImages();

    m_panel = new wxPanel(this, wxID_ANY);
    wxSize btnSize = wxSize(30,20);

    // the top sizer contains two strips:
    //     <label across the top>
    //     <grid of buttons and indicators below>
    wxBoxSizer *top_sizer = new wxBoxSizer( wxVERTICAL );
    wxBoxSizer *label_sizer = new wxBoxSizer( wxHORIZONTAL );
    wxGridSizer *grid_sizer = new wxGridSizer( 3, 3 );
    grid_sizer->SetHGap(20);    // gap, in pixels, between columns
    grid_sizer->SetVGap(20);    // gap, in pixels, between rows

    // ====================================================================
    // label group
    // ====================================================================
    // the label_sizer is arranged horizontally:
    //   <label> <spacer> <clear button>
#if 0
    wxImage img = wxImage( "src/logo.bmp", wxBITMAP_TYPE_BMP );
#else
    #include "logo.xpm"
    wxImage img = wxImage( logo_xpm );
#endif
    wxColor bg = wxSystemSettings::GetColour(wxSYS_COLOUR_BTNFACE);
    // colorize it to match background
    for(int y=0; y<img.GetHeight(); y++)
    for(int x=0; x<img.GetWidth();  x++) {
        int r = img.GetRed(x,y);
        int g = img.GetGreen(x,y);
        int b = img.GetBlue(x,y);
        img.SetRGB(x, y, int( r/255.0*bg.Red()   ),
                         int( g/255.0*bg.Green() ),
                         int( b/255.0*bg.Blue()  ) );
    }

    wxBitmap logo_bm = wxBitmap(img);
    wxStaticBitmap *logo = new wxStaticBitmap(m_panel, wxID_STATIC, logo_bm);

    wxButton *clearButton = new wxButton(m_panel, ctlId_Clear, wxT("Clear"));
    label_sizer->AddSpacer(5);
    label_sizer->Add(logo, 0, wxALIGN_LEFT | wxALL, 5);
    label_sizer->AddStretchSpacer();
    label_sizer->Add(clearButton, 0, wxALIGN_RIGHT | wxALL, 5);
    label_sizer->AddSpacer(5);

    // ====================================================================
    // grid of indicator lamps
    // ====================================================================
    // there is a 3x3 grid arranged as:
    //     Z reg,       A reg,        S reg
    //     B reg,       C reg,        M reg
    //     reg buttons, data buttons, control buttons
    for(int n=0; n<6; n++) {
        wxStaticBoxSizer *box_sizer = new wxStaticBoxSizer(wxHORIZONTAL, m_panel, groups[n]);
        for(int b=7; b>=0; b--) {
            bool status = (n == 2);
            if (status && b == 7)
                continue;       // status has no bit 7
            char **lab = (status) ? s_labels : bin_labels;
            int ctl_offset = n*8+b;
            box_sizer->Add( MakePanelLamp(lab[b], ids[n]+b, ctl_offset) );
        }
        grid_sizer->Add(box_sizer, 0, wxALIGN_CENTER);
    }

    // ------------ register select buttons ------------
    wxStaticBoxSizer *boxRegBtnSizer = new wxStaticBoxSizer(wxHORIZONTAL, m_panel, wxT("register"));
    {
        static char *reg_label[7] = { "B", "C", "Z", "A", "S", "M", "Core" };
        static int id[7] = { ctlId_regB, ctlId_regC, ctlId_regZ, ctlId_regA, ctlId_regS, ctlId_regM, ctlId_regCore };
        for(int n=0; n<7; n++)
            boxRegBtnSizer->Add( MakePanelButton(reg_label[n], id[n], btnSize) );
    }
    grid_sizer->Add(boxRegBtnSizer, 0, wxALIGN_CENTER);

    // ------------ data value buttons ------------
    m_boxDataBtns = new wxStaticBoxSizer(wxHORIZONTAL, m_panel, wxT("data"));
    for(int b=7; b>=0; b--)
        m_boxDataBtns->Add( MakePanelButton(bin_labels[b], ctlId_data+b, btnSize, true) );
    grid_sizer->Add(m_boxDataBtns, 0, wxALIGN_CENTER);

    // ------------ control/mode buttons ------------
    // there are two groups of controls in this grid space:
    //  mode (run, display, enter),  and command (load, step, exq, go)
    wxBoxSizer *grp = new wxBoxSizer( wxHORIZONTAL );
    {
        wxStaticBoxSizer *boxModeBtnSizer = new wxStaticBoxSizer(wxHORIZONTAL, m_panel, wxT("mode"));
        boxModeBtnSizer->Add( MakePanelButton("Run",     ctlId_mode_run,     btnSize, true) );
        boxModeBtnSizer->Add( MakePanelButton("Display", ctlId_mode_display, btnSize, true) );
        boxModeBtnSizer->Add( MakePanelButton("Enter",   ctlId_mode_enter,   btnSize, true) );
        grp->Add(boxModeBtnSizer);
    }

    grp->AddStretchSpacer();

    {
        wxStaticBoxSizer *boxCmdBtnSizer = new wxStaticBoxSizer(wxHORIZONTAL, m_panel, wxT("command"));
        boxCmdBtnSizer->Add( MakePanelButton("Load", ctlId_command_load, btnSize) );
        boxCmdBtnSizer->Add( MakePanelButton("Step", ctlId_command_step, btnSize) );
        boxCmdBtnSizer->Add( MakePanelButton("ExQ",  ctlId_command_exq,  btnSize) );
        boxCmdBtnSizer->Add( MakePanelButton("Go",   ctlId_command_go,   btnSize) );
        grp->Add(boxCmdBtnSizer);
    }
    grid_sizer->Add(grp, 0, wxALIGN_CENTER);

    // ====================================================================
    // glue it together and set it running
    // ====================================================================

    top_sizer->Add(label_sizer, 0, wxEXPAND | wxALL, 5);
    top_sizer->Add(grid_sizer,  0, wxEXPAND | wxALL, 5);

    m_panel->SetAutoLayout(true);
    m_panel->SetSizer(top_sizer);

    // don't allow frame to get smaller than what the sizers allow
    top_sizer->SetSizeHints(this);

    // set initial state for mode
    SetModeButtons(ctlId_mode_display);

    // intercept all keystrokes to all controls so we can do some trickery
    connectKeyDownEvent(this);

    Show(true);
}


// recursively attach a key event handler to all children windows
void
UiFrontPanel::connectKeyDownEvent(wxWindow* pclComponent)
{
    if (pclComponent) {
        // intercept KEY_DOWN
        pclComponent->Connect(wxID_ANY, wxEVT_KEY_DOWN,
                              wxKeyEventHandler(UiFrontPanel::OnKeyDown),
                              (wxObject*) NULL, this);
        // intercept KEY_UP
        pclComponent->Connect(wxID_ANY, wxEVT_KEY_UP,
                              wxKeyEventHandler(UiFrontPanel::OnKeyDown),
                              (wxObject*) NULL, this);
// FIXME: we don't receive any KEY_UP messages

        // intercept CHAR
        pclComponent->Connect(wxID_ANY, wxEVT_CHAR,
                              wxKeyEventHandler(UiFrontPanel::OnKeyDown),
                              (wxObject*) NULL, this);
// FIXME: we don't receive any CHAR messages

        wxWindowListNode* pclNode = pclComponent->GetChildren().GetFirst();
        while(pclNode) {
            wxWindow* pclChild = pclNode->GetData();
            this->connectKeyDownEvent(pclChild);
            pclNode = pclNode->GetNext();
        }
    }
}


// any time a control receives a keystroke, this function is called before
// anything happens.  if the mouse is hovering over the data buttons and
// it a hex character, the state of of the four ls buttons is moved up to
// the four msbs, and the just entered key is used to set the four ls bits.
// thus entering two hex keystrokes in a row will set the data button state.
void
UiFrontPanel::OnKeyDown(wxKeyEvent& event)
{
    bool keydown = (event.GetEventType() == wxEVT_KEY_DOWN);
    bool keyup   = (event.GetEventType() == wxEVT_KEY_UP);
    bool keychar = (event.GetEventType() == wxEVT_CHAR);

    // get rect describing box sizer around data input buttons
    wxRect rc( m_boxDataBtns->GetPosition(), m_boxDataBtns->GetSize() );

    // if the keystroke occurred while the mouse was over this rect,
    // intercept it.
    wxPoint pt(event.GetPosition());
    // this value is relative to the upper left corner of the control
    // that has focus.
    wxPoint ctlPt(FindFocus()->GetPosition());
    pt += ctlPt;

    if (rc.Contains(pt)) {
        int ch = event.GetKeyCode();
        bool isdig = (ch >= '0' && ch <= '9');
        bool ishex = (ch >= 'A' && ch <= 'F');
        if (isdig || ishex) {
            if (keydown) {
                int hex = (isdig) ? ch - '0' : ch - 'A' + 10;
                int oldv = GetDataButtons();
                int newv = ((oldv << 4) & 0xF0) | hex;
                SetDataButtons(newv);
            }
            return;     // swallow KEY_DOWN and KEY_UP
        }
        return;
    }

    // otherwise, just pass it on
    event.Skip();
} 


// save/get options to the config file
void
UiFrontPanel::SaveDefaults()
{
    wxString subgroup;
    subgroup.Printf("ui/Frontpanel");

    // save position and size
    ConfigWriteWinGeom(this, subgroup);
}

void
UiFrontPanel::GetDefaults()
{
    // pick up screen color scheme
    wxString subgroup;
    subgroup.Printf("ui/Frontpanel");

    // pick up screen location and size
    wxRect default_geom(50,50,690,380);
    ConfigReadWinGeom(this, subgroup, &default_geom);
}
